//############################################################################
//
// EZFB ~ Linux Frame Buffer API ~
//
// by James Lehman
// james@akrobiz.com
//
// began: Feb. 2001
//
// EZFB is intended to give C programmers an easy-to-use library of functions
// for drawing points, lines and strings of characters into the Linux frame
// buffer. It also provides a means of displaying bitmap images on any portion
// of the screen and capturing the contents of any portion of the screen to
// a standard bitmap graphics file. All of this is embodied in a set of
// functions that are polymorphic with regard to the characteristics of the
// frame buffer and the bitmap files used. In other words, it makes no
// difference what the screen height or width is or what the color depth is,
// all of the function calls are the same and will have appropriate and
// predictable effects.
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ezfb_device.h"
#include "ezfb_colors.h"
#include "ezfb_pixels.h"
#include "ezfb_fonts.h"
#include "ezfb_bmp.h"

//############################################################################
//  UTILITY ROUTINES
int ezfb_fix_rectangle           (struct ezfb_rectangle* area, int max_x, int max_y);
//  BITMAP IN MEMORY
int bmp_init                     (struct ezfb_bitmap* bmp, u_int x, u_int y, u_short bpp);
int bmp_malloc                   (struct ezfb_bitmap* bmp);
int bmp_find_black_and_white     (struct ezfb_bitmap* bmp);
// BITMAP MEMORY AND FILE
int bmp_open_for_read            (struct ezfb_bitmap* bmp, char* file_name);
int bmp_read_file_header         (struct ezfb_bitmap* bmp);
int bmp_read_file_palette        (struct ezfb_bitmap* bmp);
int bmp_read_file_image          (struct ezfb_bitmap* bmp);
int bmp_open_for_write           (struct ezfb_bitmap* bmp, char* file_name);
int bmp_write_file_header        (struct ezfb_bitmap* bmp);
int bmp_write_palette            (struct ezfb_bitmap* bmp);
int bmp_write_image              (struct ezfb_bitmap* bmp);
//  FRAME BUFFER OUTPUT AS BITMAP FILE
int bmp_write_palette_from_ezfb  (struct ezfb* fb, struct ezfb_bitmap* bmp);
int bmp_write_image_from_ezfb    (struct ezfb* fb, struct ezfb_bitmap* bmp, struct ezfb_rectangle* fb_area);
//  BITMAP FILE TO FRAME BUFFER
int bmp_file_palette_to_ezfb_    (struct ezfb* fb, struct ezfb_bitmap* bmp);
int bmp_file_image_to_ezfb       (struct ezfb* fb, struct ezfb_bitmap* bmp, struct ezfb_rectangle* fb_area, struct ezfb_rectangle* bmp_area);

//############################################################################
// UTILITY ROUTINES
//############################################################################
int ezfb_fix_rectangle(struct ezfb_rectangle* area, int max_x, int max_y)
{
    int swap;
EZFB_FUNCTION_CALL
    if(0 > area->x1)            area->x1 = 0;
    if(0 > area->x2)            area->x2 = max_x;
    if(0 > area->y1)            area->y1 = 0;
    if(0 > area->y2)            area->y2 = max_y;
    if(area->x1 > area->x2)  {  swap = area->x2;  area->x2 = area->x1;  area->x1 = swap;  }
    if(area->y1 > area->y2)  {  swap = area->y2;  area->y2 = area->y1;  area->y1 = swap;  }
    if(max_x < area->x2)        area->x2 = max_x;
    if(max_y < area->y2)        area->y2 = max_y;
    area->height = area->y2 - area->y1 + 1;
    area->width  = area->x2 - area->x1 + 1;
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int ezfb_dump_rectangle(struct ezfb_rectangle* area, char* message)
{
EZFB_FUNCTION_CALL
#if EZFB_API_MESSAGES
    printf("\n             %s\n"         , message      );
    printf(  "             x1     = %d\n", area->x1     );
    printf(  "             y1     = %d\n", area->y1     );
    printf(  "             x2     = %d\n", area->x2     );
    printf(  "             y2     = %d\n", area->y2     );
    printf(  "             width  = %d\n", area->width  );
    printf(  "             height = %d\n", area->height );
    printf("\n"                          );fflush(stdout);
#endif
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_dump_header(struct ezfb_bitmap* bmp, char* file_name, char* message)
{
EZFB_FUNCTION_CALL
#if EZFB_API_MESSAGES
    printf("\n%s %s\n", message, file_name);
    //printf(  "             B              = %c\n"  , bmp->B              );
    //printf(  "             M              = %c\n"  , bmp->M              );
    printf(  "             file_size      = %d\n"  , bmp->file_size      );
    //printf(  "             reserved       = %d\n"  , bmp->reserved       );
    //printf(  "             jump           = %d\n"  , bmp->jump           );
    //printf(  "             to_end         = %d\n"  , bmp->to_end         );
    printf(  "             xres           = %d\n"  , bmp->xres           );
    printf(  "             yres           = %d\n"  , bmp->yres           );
    //printf(  "             planes         = %d\n"  , bmp->planes         );
    printf(  "             bpp            = %d\n"  , bmp->bpp            );
    //printf(  "             comp           = %d\n"  , bmp->comp           );
    printf(  "             image_size     = %d\n"  , bmp->image_size     );
    printf(  "             xppm           = %d\n"  , bmp->xppm           );
    printf(  "             yppm           = %d\n"  , bmp->yppm           );
    printf(  "             c_used         = %d\n"  , bmp->c_used         );
    printf(  "             c_important    = %d\n"  , bmp->c_important    );
    printf(  "             black          = %d\n"  , bmp->black          );
    printf(  "             white          = %d\n"  , bmp->white          );
    //printf(  "             fd             = %d\n"  , bmp->fd             );
    //printf(  "             bytes_per_line = %d\n"  , bmp->bytes_per_line );
    printf(  "             palette_size   = %d\n"  , bmp->palette_size   );
    printf(  "\n"                                             );fflush(stdout);
#endif
EZFB_FUNCTION_RETURN(1)
}

//############################################################################















//############################################################################
// BITMAP IN MEMORY
//############################################################################
int bmp_create(struct ezfb_bitmap* bmp, u_int x, u_int y, u_short bpp)
{
EZFB_FUNCTION_CALL
	bmp_init   (bmp, x, y, bpp);
	bmp_malloc (bmp);
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_clone_ezfb_screen(struct ezfb* fb, struct ezfb_bitmap* bmp, int copy_image)
{    // Only works with 1, 4, 8, 24 bpp !!!
EZFB_FUNCTION_CALL
	bmp_create(bmp, fb->Var.xres, fb->Var.yres, fb->Var.bits_per_pixel);
	bmp_palette_from_ezfb(fb, bmp);
	if(copy_image)
    {
        for(u_int x = 0; x < fb->Var.xres; x++)
            for(u_int y = 0; y < fb->Var.yres; y++)
                bmp_put_pixel(bmp, x, y, ezfb_get_pixel(fb, x, y));
    }
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_clone_ezfb_virtual(struct ezfb* fb, struct ezfb_bitmap* bmp, int copy_image)
{    // Only works with 1, 4, 8, 24 bpp !!!
EZFB_FUNCTION_CALL
	bmp_create(bmp, fb->Var.xres_virtual, fb->Var.yres_virtual, fb->Var.bits_per_pixel);
	bmp_palette_from_ezfb(fb, bmp);
	if(copy_image)
		memcpy((void*)(bmp->image), (const void*)(fb->screen_uchars), bmp->image_size);
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_init(struct ezfb_bitmap* bmp, u_int x, u_int y, u_short bpp)
{
EZFB_FUNCTION_CALL
    bmp->B          = 'B'                          ;
    bmp->M          = 'M'                          ;
    bmp->reserved   = 0                            ;
    bmp->to_end     = EZFB_BMP_FILE_HEADER_SIZE - 14 ;
    bmp->xres       = x                            ;
    bmp->yres       = y                            ;
    bmp->planes     = 1                            ;
    bmp->comp       = 0                            ;
    bmp->xppm       = 2835                         ;
    bmp->yppm       = 2835                         ;
    bmp->black      = 0                            ;
    bmp->palette    = (u_int* )NULL                ;
    bmp->image      = (u_char*)NULL                ;
    bmp->data       = (u_char*)NULL                ;
    switch(bpp)
    {
        case  1: bmp->bytes_per_line = (int)(ceil((bmp->xres * 0.125) / 4.0) * 4);
                 bmp->bpp            =   1;
                 bmp->palette_size   =   8;
                 bmp->c_used         =   2;
                 bmp->c_important    =   2;
                 bmp->white          =   1;
                 break;
        // -------------------------------------------------------------
        case  2:
        case  4: bmp->bytes_per_line = (int)(ceil((bmp->xres * 0.5) / 4.0) * 4);
                 bmp->bpp            =   4;
                 bmp->palette_size   =  64;
                 bmp->c_used         =  16;
                 bmp->c_important    =  16;
                 bmp->white          =  15;
                 break;
        // -------------------------------------------------------------
        case  8: bmp->bytes_per_line = (int)(ceil( bmp->xres / 4.0) * 4);
                 bmp->bpp            =    8;
                 bmp->palette_size   = 1024;
                 bmp->c_used         =  256;
                 bmp->c_important    =  256;
                 bmp->white          =  255;
                 break;
        // -------------------------------------------------------------
        case 15:
        case 16: 
        case 24:
        case 32: bmp->bytes_per_line = (int)(ceil((bmp->xres * 3.0) / 4.0) * 4);
                 bmp->bpp            =         24;
                 bmp->palette_size   =          0;
                 bmp->c_used         =          0;
                 bmp->c_important    =          0;
                 bmp->white          = 0xffffffff;
                 break;
        // -------------------------------------------------------------
        default: bmp->bytes_per_line =   0;
                 bmp->bpp            =   0;
                 bmp->palette_size   =   0;
                 bmp->c_used         =   0;
                 bmp->c_important    =   0;
                 bmp->white          =   0;
                 break;
        // -------------------------------------------------------------
    }
    bmp->image_size = bmp->yres * bmp->bytes_per_line             ;
    bmp->jump       = EZFB_BMP_FILE_HEADER_SIZE + bmp->palette_size ;
    bmp->file_size  = bmp->image_size + bmp->jump                 ;
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_malloc(struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL
    if(bmp->data) free(bmp->data);
    bmp->data    = (u_char*)malloc(bmp->palette_size + bmp->image_size);
    bmp->palette = (u_int*)bmp->data;
    bmp->image   = bmp->data + bmp->palette_size;
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_put_pixel(struct ezfb_bitmap* bmp, u_int x, u_int y, u_int c)
{
    if(bmp->data && (x < bmp->xres) && (y < bmp->yres))
    {
        switch(bmp->bpp)
        {
            case  1: break;
            /* ----------------------------------------------------------------------------- */
            case  4: {
                         int i = (y * bmp->xres + x) / 2;
                         bmp->image[i] =   (x & 0x01) // odd
                                         ? ((bmp->image[i] & 0x0F) | ((c & 0x0F) << 4))
                                         : ((bmp->image[i] & 0xF0) |  (c & 0x0F)      );
                     }
                     break;
            /* ----------------------------------------------------------------------------- */
            case  8: bmp->image[y * bmp->xres + x] = (u_char)c;
                     break;
            /* ----------------------------------------------------------------------------- */
            case 24: {
                         int i = (y * bmp->xres + x) * 3;
                         bmp->image[i++] = c & 0x000000ff;
                         bmp->image[i++] = c & 0x0000ff00;
                         bmp->image[i  ] = c & 0x00ff0000;
                     }
                     break;
            /* ----------------------------------------------------------------------------- */
            default: return 0;
        }
        return 1;
    }
    return 0;
}

//############################################################################
int bmp_get_pixel(struct ezfb_bitmap* bmp, u_int x, u_int y)
{
    if(bmp->data && (x < bmp->xres) && (y < bmp->yres))
    {
        switch(bmp->bpp)
        {
            case  1: return 0;
            /* ----------------------------------------------------------------------------- */
            case  4: {
                        int i = (y * bmp->xres + x) / 2;
                        return (u_int)( (x & 0x01) // odd
                                          ? ((bmp->image[i] & 0xF0) >> 4)
                                          :  (bmp->image[i] & 0x0F)       );
                     }
            /* ----------------------------------------------------------------------------- */
            case  8: return (u_int)bmp->image[y * bmp->xres + x];
            /* ----------------------------------------------------------------------------- */
            case 24: {
                         int i = (y * bmp->xres + x) * 3;
                         return   (bmp->image[i++] << 16)
                                | (bmp->image[i++] <<  8)
                                |  bmp->image[i  ]        ;
                     }
            /* ----------------------------------------------------------------------------- */
            default: return -1;
        }
    }
    return -1;
}

//############################################################################
int bmp_find_black_and_white(struct ezfb_bitmap* bmp)
{
    if(bmp->palette)
    {
        switch(bmp->bpp)
        {
            case  1:  bmp->black = 0;
                      bmp->white = 1;
                      break;
            //----------------------------------------------------------------
            case  2:
            case  4:
            case  8: {
                          u_int   i;
                          bmp->black = 0;
                          bmp->white = 0;
                          for(i = 0; i < bmp->c_used; i++)
                          {
                              if(   (    ((bmp->palette[bmp->black] & 0x0000ff00) >>  8)
                                       + ((bmp->palette[bmp->black] & 0x00ff0000) >> 16)
                                       + ((bmp->palette[bmp->black] & 0xff000000) >> 24)
                                    )
                                  > (    ((bmp->palette[i] & 0x0000ff00) >>  8)
                                       + ((bmp->palette[i] & 0x00ff0000) >> 16)
                                       + ((bmp->palette[i] & 0xff000000) >> 24)
                                    )
                                )
                                  bmp->black = i;
                              //----------------------------------------------
                              if(   (    ((bmp->palette[bmp->white] & 0x0000ff00) >>  8)
                                       + ((bmp->palette[bmp->white] & 0x00ff0000) >> 16)
                                       + ((bmp->palette[bmp->white] & 0xff000000) >> 24)
                                    )
                                  < (    ((bmp->palette[i] & 0x0000ff00) >>  8)
                                       + ((bmp->palette[i] & 0x00ff0000) >> 16)
                                       + ((bmp->palette[i] & 0xff000000) >> 24)
                                    )
                                )
                                  bmp->white = i;
                          }
                      }
                      break;
            //----------------------------------------------------------------
            case 24:
            default:  bmp->black = 0x00000000;
                      bmp->white = 0xffffffff;
                      break;
        }
        return 1;
    }
    return 0;
}

//############################################################################
long double bmp_put_line(struct ezfb_bitmap* bmp, int x1, int y1, int x2, int y2, u_int c)
{
    int dx = x2 - x1;
    int dy = y2 - y1;
    int n, b;
    long double m;

    if(0 == dx)
    {
        if(dy < 0)
            for(n = y1; n >= y2; n--)
                bmp_put_pixel(bmp, x1, n, c);
        else
            for(n = y1; n <= y2; n++)
                bmp_put_pixel(bmp, x1, n, c);
    }
    else if(0 == dy)
    {
        if(dx < 0)
            for(n = x1; n >= x2; n--)
                bmp_put_pixel(bmp, n, y1, c);
        else
            for(n = x1; n <= x2; n++)
                bmp_put_pixel(bmp, n, y1, c);
    }
    else if(abs(dx) >= abs(dy))
    {
        m = ((long double)dy) / ((long double)dx);
        b = (int)(y1 - m * x1);
        if(dx < 0)
            for(n = x1; n >= x2; n--)
                bmp_put_pixel(bmp, n, (int)(n * m) + b, c);
        else
            for(n = x1; n <= x2; n++)
                bmp_put_pixel(bmp, n, (int)(n * m) + b, c);
    }
    else
    {
        m = ((long double)dx) / ((long double)dy);
        b = (int)(x1 - m * y1);
        if(dy < 0)
            for(n = y1; n >= y2; n--)
                bmp_put_pixel(bmp, (int)(n * m) + b, n, c);
        else
            for(n = y1; n <= y2; n++)
                bmp_put_pixel(bmp, (int)(n * m) + b, n, c);
    }
    return sqrt((dx * dx) + (dy * dy)); // magnitude of line
}

//############################################################################
long double bmp_put_ray(struct ezfb_bitmap* bmp, uint x, uint y, long double r, long double t, uint c)
{
    return bmp_put_line(bmp, x, y, (int)(x + r * cos(t)), (int)(y + r * sin(t)), c);
}

//############################################################################
void bmp_put_rectangle(struct ezfb_bitmap* bmp, int x1, int y1, int x2, int y2, u_int c, int filled)
{
    if(filled)
    {
        int y, step;
        step = (y1 <= y2) ? 1 : -1;
        if((x1 != x2) && (y1 != y2))
            for(y = y1; y <= y2; y += step)
                bmp_put_line(bmp, x1, y, x2, y, c);
        else if(x1 != x2)
            bmp_put_line(bmp, x1, y1, x2, y1, c);
        else
            bmp_put_pixel(bmp, x1, y1, c);
    }
    else
    {
        if((x1 != x2) && (y1 != y2))
        {
            bmp_put_line(bmp, x1, y1, x2, y1, c);
            bmp_put_line(bmp, x1, y2, x2, y2, c);
            bmp_put_line(bmp, x1, y1, x1, y2, c);
            bmp_put_line(bmp, x2, y1, x2, y2, c);
        }
        else if(x1 != x2)
            bmp_put_line(bmp, x1, y1, x2, y1, c);
        else
            bmp_put_pixel(bmp, x1, y1, c);
    }
	return;
}

//############################################################################
int bmp_put_byte_pattern(struct ezfb_bitmap* bmp, char byte, int x, int y, int fc, int bc, int dir, int size)
{
    int i, j, ret = 0;
    switch(dir)
    {
        default:
        case  0:
        case  2:    if(0 > bc)
                        for(i = 0; i < 8; i++)
                            for(j = 0; j < size; j++)
                                ret = bmp_put_pixel(bmp, x + i * size + j, y, (byte & (1 << (7 - i))) ? fc : bmp_get_pixel(bmp, x + i * size + j, y));

                    else if(0 > fc)
                        for(i = 0; i < 8; i++)
                            for(j = 0; j < size; j++)
                                ret = bmp_put_pixel(bmp, x + i * size + j, y, (byte & (1 << (7 - i))) ? bmp_get_pixel(bmp, x + i * size + j, y) : bc);

                    else
                        for(i = 0; i < 8; i++)
                            for(j = 0; j < size; j++)
                                ret = bmp_put_pixel(bmp, x + i * size + j, y, (byte & (1 << (7 - i))) ? fc : bc);

                    return x + 8 * size * ret;
        // ----------------------------------------------------------------------
        case  1:
        case  3:    if(0 > bc)
                        for(i = 0; i < 8; i++)
                            for(j = 0; j < size; j++)
                                ret = bmp_put_pixel(bmp, x, y + i * size + j, (byte & (1 << (7 - i))) ? fc : bmp_get_pixel(bmp, x, y + i * size + j));

                    else if(0 > fc)
                        for(i = 0; i < 8; i++)
                            for(j = 0; j < size; j++)
                                ret = bmp_put_pixel(bmp, x, y + i * size + j, (byte & (1 << (7 - i))) ? bmp_get_pixel(bmp, x, y + i * size + j) : bc);

                    else
                        for(i = 0; i < 8; i++)
                            for(j = 0; j < size; j++)
                                ret = bmp_put_pixel(bmp, x, y + i * size + j, (byte & (1 << (7 - i))) ? fc : bc);

                    return y + 8 * size * ret;
        // ----------------------------------------------------------------------
    }
}

//############################################################################
int bmp_put_ascii(struct ezfb_bitmap* bmp, char ascii, int x, int y, int fc, int bc, int dir, int size)
{
    int i, j, ret = 0;
    switch(dir)
    {
        default:
        case  0:    for(i = 0; i < 8; i++)
                        for(j = 0; j < size; j++)
                            ret = bmp_put_byte_pattern(bmp, ezfb_fontdata_8x8[ascii * 8 + i], x, y + i * size + j, fc, bc, dir, size);
                    break;
        // ----------------------------------------------------------------------
        case  1:    for(i = 0; i < 8; i++)
                        for(j = 0; j < size; j++)
                            ret = bmp_put_byte_pattern(bmp, ezfb_fontdata_8x8[ascii * 8 + (7 - i)], x + i * size + j, y, fc, bc, dir, size);
                    break;
        // ----------------------------------------------------------------------
        case  2:    for(i = 0; i < 8; i++)
                        for(j = 0; j < size; j++)
                            ret = bmp_put_byte_pattern(bmp, ezfb_fontdata_8x8[ascii * 8 + i], x, y - i * size - j, fc, bc, dir, size);
                    break;
        // ----------------------------------------------------------------------
        case  3:    for(i = 0; i < 8; i++)
                        for(j = 0; j < size; j++)
                            ret = bmp_put_byte_pattern(bmp, ezfb_fontdata_8x8[ascii * 8 + i], x + i * size + j, y, fc, bc, dir, size);
                    break;
        // ----------------------------------------------------------------------
    }
    return ret;
}

//############################################################################
int bmp_put_string(struct ezfb_bitmap* bmp, char* str, int x, int y, int fc, int bc, int dir, int size)
{
	u_int i;
	int   j, ret = 0;
	switch(dir)
	{
		default:
		case  0:    for(i = 0; i < strlen(str); i++)
						for(j = 0; j < size; j++)
							ret = bmp_put_ascii(bmp, str[i], x + i * 8 * size, y + j, fc, bc, dir, size);
					break;
		// ----------------------------------------------------------------------
		case  1:    for(i = 0; i < strlen(str); i++)
						for(j = 0; j < size; j++)
							ret = bmp_put_ascii(bmp, str[i], x + j, y + i * 8 * size, fc, bc, dir, size);
					break;
		// ----------------------------------------------------------------------
		case  2:    for(i = 0; i < strlen(str); i++)
						 for(j = 0; j < size; j++)
						   ret = bmp_put_ascii(bmp, str[i], x - i * 8 * size, y + j, fc, bc, dir, size);
					break;
		// ----------------------------------------------------------------------
		case  3:    for(i = 0; i < strlen(str); i++)
						for(j = 0; j < size; j++)
							ret = bmp_put_ascii(bmp, str[i], x + j, y - i * 8 * size, fc, bc, dir, size);
					break;
		// ----------------------------------------------------------------------
	}
	return ret;
}

//############################################################################
int bmp_printf(struct ezfb_bitmap* bmp, int x, int y, int fc, int bc, int dir, int size, const char* text_format, ...)
{
    char formatted_text[1024];
    va_list var_arg_list;
    fflush(stdout);
    va_start(var_arg_list, text_format);
    vsprintf(formatted_text, text_format, var_arg_list);
    va_end(var_arg_list);
    return bmp_put_string(bmp, formatted_text, x, y, fc, bc, dir, size);
}

//############################################################################
int bmp_clear(struct ezfb_bitmap* bmp, u_char c)
{
EZFB_FUNCTION_CALL
    memset((void*)bmp->image, c, bmp->image_size);
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int bmp_free(struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL
    if(bmp->data)
    {
        free(bmp->data);
        bmp->data    = NULL;
        bmp->palette = NULL;
        bmp->image   = NULL;
    }
EZFB_FUNCTION_RETURN(1)
}

//############################################################################




















//############################################################################
// BITMAP MEMORY AND FILE
//############################################################################
int bmp_from_file(struct ezfb_bitmap* bmp, char* file_name)
{
EZFB_FUNCTION_CALL_1
    ret &= bmp_open_for_read     (bmp, file_name);
    ret &= bmp_read_file_header  (bmp           );
    ret &= bmp_malloc            (bmp           );
    ret &= bmp_read_file_palette (bmp           );
    ret &= bmp_read_file_image   (bmp           );
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_open_for_read(struct ezfb_bitmap* bmp, char* file_name)
{
EZFB_FUNCTION_CALL_1
    if(0 > (bmp->fd = open(file_name, O_RDONLY))) // open the bitmap file
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: opening bmp file failed");
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_read_file_header(struct ezfb_bitmap* bmp)
{
    int bytes_read = 0;
EZFB_FUNCTION_CALL_1
    // file descriptor must be open, valid and in bmp->fd
    lseek(bmp->fd, 0, SEEK_SET); // start at the top
    bytes_read += read(bmp->fd, (void*)(&bmp->B           ), sizeof(char) );
    bytes_read += read(bmp->fd, (void*)(&bmp->M           ), sizeof(char) );
    bytes_read += read(bmp->fd, (void*)(&bmp->file_size   ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->reserved    ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->jump        ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->to_end      ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->xres        ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->yres        ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->planes      ), sizeof(short));
    bytes_read += read(bmp->fd, (void*)(&bmp->bpp         ), sizeof(short));
    bytes_read += read(bmp->fd, (void*)(&bmp->comp        ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->image_size  ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->xppm        ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->yppm        ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->c_used      ), sizeof(int)  );
    bytes_read += read(bmp->fd, (void*)(&bmp->c_important ), sizeof(int)  );

    bmp->bytes_per_line = (int)(ceil((bmp->bpp / 8.0) * bmp->xres / 4.0) * 4);
    bmp->palette_size   = bmp->jump - EZFB_BMP_FILE_HEADER_SIZE;
    bmp->black          = 0;
    bmp->white          = 0;

    if(bytes_read != EZFB_BMP_FILE_HEADER_SIZE)
        ret = 0;

EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_read_file_palette(struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL_0
    if(bmp->data && bmp->bpp <= 8)
    {
        lseek(bmp->fd, 14 + bmp->to_end, SEEK_SET);
        ret = (    bmp->palette_size
                == (u_int)read(bmp->fd, (void*)(bmp->palette), bmp->palette_size));
        bmp_find_black_and_white(bmp);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_read_file_image(struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL_0
    if(bmp->data)
    {
        u_int bytes_read = 0,
            i;
        lseek(bmp->fd, bmp->jump, SEEK_SET);
        switch(bmp->bpp)
        {
            case  1: for(i = 0; i < bmp->yres; i++)
                     {
                         bytes_read += read(bmp->fd, (void*)(bmp->image), bmp->xres / 8);
                         lseek(bmp->fd, bmp->bytes_per_line - bmp->xres / 8, SEEK_CUR);
                     }
                     break;
            // ---------------------------------------------------------------
            case  4: for(i = 0; i < bmp->yres; i++)
                     {
                         bytes_read += read(bmp->fd, (void*)(bmp->image), bmp->xres / 2);
                         lseek(bmp->fd, bmp->bytes_per_line - bmp->xres / 2, SEEK_CUR);
                     }
                     break;
            // ---------------------------------------------------------------
            case  8: for(i = 0; i < bmp->yres; i++)
                     {
                         bytes_read += read(bmp->fd, (void*)(bmp->image), bmp->xres);
                         lseek(bmp->fd, bmp->bytes_per_line - bmp->xres, SEEK_CUR);
                     }
                     break;
            // ---------------------------------------------------------------
            case 24: for(i = 0; i < bmp->yres; i++)
                     {
                         bytes_read += read(bmp->fd, (void*)(bmp->image), bmp->xres * 3);
                         lseek(bmp->fd, bmp->bytes_per_line - bmp->xres * 3, SEEK_CUR);
                     }
                     break;
            // ---------------------------------------------------------------
        }
        ret = (bmp->image_size == bytes_read);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_open_for_write(struct ezfb_bitmap* bmp, char* file_name)
{
EZFB_FUNCTION_CALL_1
    if(0 > (bmp->fd = open(file_name, (O_WRONLY | O_CREAT), (S_IRWXU | S_IRGRP | S_IROTH)))) // mode 744
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: opening bmp file failed");
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_write_file_header(struct ezfb_bitmap* bmp)
{
    int bytes_written = 0;
EZFB_FUNCTION_CALL_1
    if(bmp->fd)
    {
        bytes_written += write(bmp->fd, (void*)(&bmp->B           ), sizeof(char) );
        bytes_written += write(bmp->fd, (void*)(&bmp->M           ), sizeof(char) );
        bytes_written += write(bmp->fd, (void*)(&bmp->file_size   ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->reserved    ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->jump        ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->to_end      ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->xres        ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->yres        ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->planes      ), sizeof(short));
        bytes_written += write(bmp->fd, (void*)(&bmp->bpp         ), sizeof(short));
        bytes_written += write(bmp->fd, (void*)(&bmp->comp        ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->image_size  ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->xppm        ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->yppm        ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->c_used      ), sizeof(int)  );
        bytes_written += write(bmp->fd, (void*)(&bmp->c_important ), sizeof(int)  );
    }
    if(bytes_written != (14 + bmp->to_end))
        ret = 0;
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_write_palette(struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL_0
    ret = (    bmp->palette_size
            == (u_int)write(bmp->fd, (void*)(bmp->palette), bmp->palette_size));
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_write_image(struct ezfb_bitmap* bmp)
{
    u_int i,
          bytes_written  = 0,
          zero           = 0;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < bmp->yres; i++)
    {
        switch(bmp->bpp)
        {
            case  1: break;
            // ---------------------------------------------------------------
            case  4: {
                         u_int   j,
                                 local_x;
                         u_char  c;
                         for(local_x = 0; local_x < bmp->xres / 2; local_x++)
                         {
                             j = i * bmp->xres / 2 + local_x;
                             c = ((bmp->image[j] & 0x0F) << 4) + ((bmp->image[j] & 0xF0) >> 4); // nibble swap!
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));
                         }
                         bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - bmp->xres / 2));
                     }
                     break;
            // ---------------------------------------------------------------
            case  8: bytes_written += write(bmp->fd, (void*)&(bmp->image[i * bmp->xres]), bmp->bytes_per_line);
                     bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - bmp->xres));
                     break;
            // ---------------------------------------------------------------
            case 24: bytes_written += write(bmp->fd, (void*)&(bmp->image[i * bmp->xres * 3]), bmp->bytes_per_line);
                     bytes_written += write(bmp->fd, (void*)&(zero), bmp->bytes_per_line - bmp->xres * 3);
                     break;
            // ---------------------------------------------------------------
        }
    }
    ret = (bytes_written == bmp->image_size);
#if EZFB_API_MESSAGES
    printf("\n****** bytes_written = %d ******\n", bytes_written);fflush(stdout);
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_save_as(struct ezfb_bitmap* bmp, char* file_name)
{
EZFB_FUNCTION_CALL_1
    ret &= bmp_open_for_write (bmp, file_name);
    ret &= bmp_write_palette  (bmp);
    ret &= bmp_write_image    (bmp);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################




















//############################################################################
// FRAME BUFFER OUTPUT AS BMP FILE ROUTINES
//############################################################################
int bmp_write_palette_from_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp)
{
    u_char c;
    u_int  i,
           bytes_written = 0;
EZFB_FUNCTION_CALL_1
    // file descriptor must be open, valid and in bmp->fd
    if(bmp->bpp != 24)
    {
        for(i = 0; i < bmp->c_used; i++) // must be called after bmp_write_file_header(...
        {
            c = ((fb->colors.blue  [i] >> (fb->Var.blue.msb_right ? 0 : EZFB_BITS_IN_CHAR)) & fb->colors.bit_mask_blue);
            bytes_written += write(bmp->fd, (void*)(&c), sizeof(char));

            c = ((fb->colors.green [i] >> (fb->Var.green.msb_right ? 0 : EZFB_BITS_IN_CHAR)) & fb->colors.bit_mask_green);
            bytes_written += write(bmp->fd, (void*)(&c), sizeof(char));

            c = ((fb->colors.red   [i] >> (fb->Var.red.msb_right ? 0 : EZFB_BITS_IN_CHAR)) & fb->colors.bit_mask_red);
            bytes_written += write(bmp->fd, (void*)(&c), sizeof(char));

            c = (char)0;
            bytes_written += write(bmp->fd, (void*)(&c), sizeof(char));
        }
        ret = (bytes_written == bmp->palette_size);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_write_image_from_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp, struct ezfb_rectangle* fb_area)
{
    int   i;
    u_int bytes_written  = 0,
          zero           = 0;
EZFB_FUNCTION_CALL_0
    // must be called after bmp_write_file_header(... and bmp_write_palette_from_ezfb(...
    for(i = fb_area->y2; i >= fb_area->y1; i--)
    {
        switch(fb->Var.bits_per_pixel)
        {
            case  1: break;
            // ---------------------------------------------------------------
            case  2: break;
            // ---------------------------------------------------------------
            case  4: {
                         int   j,
                               local_x;
                         u_char  c;
                         for(local_x = fb_area->x1; local_x <= (fb_area->x2 / 2); local_x++)
                         {
                             j = i * fb->Var.xres_virtual / 2 + local_x;
                             c = ((fb->screen_uchars[j] & 0x0F) << 4) + ((fb->screen_uchars[j] & 0xF0) >> 4); // nibble swap!
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));
                         }
                         bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - fb_area->width / 2));
                     }
                     break;
            // ---------------------------------------------------------------
            default:
            case  8: bytes_written += write(bmp->fd, (void*)&(fb->screen_uchars[i * fb->Var.xres_virtual + fb_area->x1]), fb_area->width);
                     bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - fb_area->width));
                     break;
            // ---------------------------------------------------------------
            case 15:
            case 16: {
                         int   j,
                               local_x;
                         u_char  c;
                         for(local_x = fb_area->x1; local_x <= fb_area->x2; local_x++)
                         {
                             j = fb->screen_ushorts[i * fb->Var.xres_virtual + local_x];

                             c = ezfb_get_b_from_rgb(fb, j);
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));

                             c = ezfb_get_g_from_rgb(fb, j);
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));

                             c = ezfb_get_r_from_rgb(fb, j);
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));
                         }
                         bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - (fb_area->width * 3)));
                     }
                     break;
            // ---------------------------------------------------------------
            case 24: bytes_written += write(bmp->fd, (void*)&(fb->screen_uchars[(i * fb->Var.xres_virtual + fb_area->x1) * 3]), fb_area->width * 3);
                     bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - (fb_area->width * 3)));
                     break;
            // ---------------------------------------------------------------
            case 32: {
                         int   j,
                               local_x;
                         u_char  c;
                         for(local_x = fb_area->x1; local_x <= fb_area->x2; local_x++)
                         {
                             j = fb->screen_uints[i * fb->Var.xres_virtual + local_x];

                             c = (char)ezfb_get_b_from_rgb(fb, j);
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));

                             c = (char)ezfb_get_g_from_rgb(fb, j);
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));

                             c = (char)ezfb_get_r_from_rgb(fb, j);
                             bytes_written += write(bmp->fd, (void*)&c, sizeof(char));
                         }
                         bytes_written += write(bmp->fd, (void*)&(zero), (bmp->bytes_per_line - (fb_area->width * 3)));
                     }
                     break;
            // ---------------------------------------------------------------
        }
    }
    ret = (bytes_written == bmp->image_size);
#if EZFB_API_MESSAGES
    printf("\n****** bytes_written = %d ******\n", bytes_written);fflush(stdout);
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_save_ezfb_as(struct ezfb* fb, char* file_name)
{
    struct ezfb_rectangle fb_area = { -1, -1, -1, -1, 0, 0 };
EZFB_FUNCTION_CALL_0
    ret = bmp_save_ezfb_area_as(fb, file_name, &fb_area);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_save_ezfb_area_as(struct ezfb* fb, char* file_name, struct ezfb_rectangle* fb_area)
{
    struct ezfb_bitmap  bmp;
EZFB_FUNCTION_CALL_0
    if(!fb->exists)
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: (bmp_save_ezfb_area_as) FB NOT INITIALIZED");fflush(stderr);
#endif
    }
    else if(bmp_open_for_write(&bmp, file_name))
    {
        ezfb_fix_rectangle(fb_area, fb->max_x, fb->max_y);
#ifdef EZFB_API_MESSAGES
        ezfb_dump_rectangle(fb_area, "fb_area");
#endif
        ret  = bmp_init              (&bmp, fb->Var.xres, fb->Var.yres, fb->Var.bits_per_pixel);
        ret &= bmp_write_file_header (&bmp);
#ifdef EZFB_API_MESSAGES
        bmp_dump_header              (&bmp, file_name, "SAVING .....");
#endif
        if(bmp.bpp != 24)
            ret &= bmp_write_palette_from_ezfb(fb, &bmp);
        ret &= bmp_write_image_from_ezfb(fb, &bmp, fb_area);
        close(bmp.fd);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################




















//############################################################################
// FRAME BUFFER AND BITMAP IN MEMORY
//############################################################################
int bmp_to_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp)
{
    u_int x, y, xmax, ymax;
EZFB_FUNCTION_CALL_1
	if((fb->Var.xres_virtual == bmp->xres) && (fb->Var.yres_virtual == bmp->yres))
	{
		if((ret = ezfb_set_bpp(fb, bmp->bpp)))
			memcpy((void*)(fb->screen_uchars), (const void*)(bmp->image), bmp->image_size);
	}
    else
	{
		xmax = (fb->Var.xres > bmp->xres) ? (bmp->xres) : (fb->Var.xres);
		ymax = (fb->Var.yres > bmp->yres) ? (bmp->yres) : (fb->Var.yres);
		if((ret = ezfb_set_bpp(fb, bmp->bpp)))
		{
			bmp_palette_to_ezfb(fb, bmp);
			for(y = 0; y < ymax; y++)
				for(x = 0; x < xmax; x++)
					ezfb_put_pixel(fb, x, y, bmp_get_pixel(bmp, x, y));
		}
	}
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_from_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp)
{
    u_int x, y;
EZFB_FUNCTION_CALL_1
    bmp_init   (bmp, fb->Var.xres, fb->Var.yres, fb->Var.bits_per_pixel);
    bmp_malloc (bmp);
    bmp_palette_from_ezfb(fb, bmp);
    for(y = 0; y < fb->max_y; y++)
        for(x = 0; x < fb->max_x; x++)
            bmp_put_pixel(bmp, x, y, ezfb_get_pixel(fb, x, y));
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_palette_from_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL_1
    if(bmp->bpp <= 8)
    {
        int  i;
        u_short r, g, b;
        if(!bmp->data)
            bmp_malloc(bmp);
        for(i = 0; i < (1 << bmp->bpp); i++)
        {
            ezfb_get_cmap_index(fb, i, &r, &g, &b);
            bmp->palette[i] = r | (g << 8) | (b << 16);
        }
        bmp_find_black_and_white(bmp);
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: (bmp_palette_from_ezfb) BITMAP HAS NO PALETTE");fflush(stderr);
#endif
        ret = 0;
    }    
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_palette_to_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp)
{
EZFB_FUNCTION_CALL_1
    if(bmp->bpp <= 8)
    {
        int  i;
        ezfb_set_bpp(fb, bmp->bpp);
        for(i = 0; i < (1 << bmp->bpp); i++)
        {
            ezfb_set_cmap_index(fb, i, (u_short)( bmp->palette[i] & 0x0000FF00)
                                     , (u_short)((bmp->palette[i] & 0x00FF0000) >>  8)
                                     , (u_short)((bmp->palette[i] & 0xFF000000) >> 16));
        }
        if(fb->Var.bits_per_pixel <= 8)
        {
            ret &= ezfb_set_cmap(fb);
            ret &= ezfb_find_black_and_white(fb);
        }
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: (ezfb_cmap_from_bitmap) NO BITMAP PALETTE");fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################




















//############################################################################
// BITMAP FILE TO FRAME BUFFER
//############################################################################
int bmp_file_palette_to_ezfb(struct ezfb* fb, char* file_name)
{
    struct ezfb_bitmap  bmp;
EZFB_FUNCTION_CALL_0
    if(!fb->exists)
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: (bmp_file_palette_to_ezfb) FB NOT INITIALIZED");fflush(stderr);
#endif
    }
    else if(bmp_open_for_read(&bmp, file_name)) // open the bitmap file
    {
        ret = bmp_read_file_header(&bmp);
#ifdef EZFB_API_MESSAGES
        bmp_dump_header(&bmp, file_name, "OPENING .... ");
#endif
        ret &= bmp_file_palette_to_ezfb_(fb, &bmp);
        close(bmp.fd);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_file_palette_to_ezfb_(struct ezfb* fb, struct ezfb_bitmap* bmp)
{
    u_int bytes_read = 0;
EZFB_FUNCTION_CALL_1
    // file descriptor must be open, valid and in bmp->fd
    if(bmp->bpp <= 8)
    {
        char r, g, b, a;
        int  i;
        lseek(bmp->fd, 14 + bmp->to_end, SEEK_SET); // jump to end of header / beginning of palette
        for(i = 0; i < (1 << bmp->bpp); i++)
        {
            bytes_read += read(bmp->fd, (void*)(&b), sizeof(char));
            bytes_read += read(bmp->fd, (void*)(&g), sizeof(char));
            bytes_read += read(bmp->fd, (void*)(&r), sizeof(char));
            bytes_read += read(bmp->fd, (void*)(&a), sizeof(char));
            ezfb_set_cmap_index (fb, i, r << EZFB_BITS_IN_CHAR,
                                        g << EZFB_BITS_IN_CHAR,
                                        b << EZFB_BITS_IN_CHAR);
        }
        if(bytes_read == (u_int)((1 << bmp->bpp) * 4))
        {
            if(fb->Var.bits_per_pixel >= bmp->bpp)
                ret &= ezfb_set_cmap(fb);
        }
        else
            ret = 0;
    }
    ezfb_find_black_and_white(fb);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_file_image_to_ezfb(struct ezfb* fb, struct ezfb_bitmap* bmp, struct ezfb_rectangle* fb_area, struct ezfb_rectangle* bmp_area)
{
    u_int i,
          junk_before,
          junk_after,
          bytes_read = 0;

EZFB_FUNCTION_CALL_0

    ezfb_fix_rectangle(fb_area , fb->max_x                    , fb->max_y                     );
    ezfb_fix_rectangle(bmp_area, bmp->xres - 1                , bmp->yres - 1                 );
    ezfb_fix_rectangle(bmp_area, bmp_area->x1 + fb_area->width, bmp_area->y1 + fb_area->height);
    junk_before = (int)(bmp_area->x1 * (bmp->bpp / 8.0));
    junk_after  = (int)(bmp->bytes_per_line - ((bmp_area->x1 + bmp_area->width) * (bmp->bpp / 8.0)));
    lseek(bmp->fd,  bmp->jump, SEEK_SET); // jump over the header & palette to the image
    lseek(bmp->fd, (bmp->yres - bmp_area->y2 - 1) * bmp->bytes_per_line, SEEK_CUR);
	for(i = (fb_area->y1 + bmp_area->height - 1); (int)(i) >= fb_area->y1; i--)    
    {
        switch(fb->Var.bits_per_pixel)
        {
            case  1:
            case  2: break;
            // ---------------------------------------------------------------
            case  4: switch(bmp->bpp)
                     {
                        case  1: {
                                     int     local_x;
                                     u_char  c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x += 8)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x    , i, (c >> 7) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 1, i, (c >> 6) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 2, i, (c >> 5) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 3, i, (c >> 4) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 4, i, (c >> 3) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 5, i, (c >> 2) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 6, i, (c >> 1) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 7, i, (c     ) & 1);
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case  4: {
                                     int     local_x;
                                     u_char  c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x += 2)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x    , i, (c & 0xF0) >> 4);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 1, i,  c & 0x0F      );
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case  8: {
                                     int    local_x;
                                     u_char c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x++)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x, i, ezfb_get_4bit_from_rgb(   ezfb_get_r_from_cmap_index(fb, c)
                                                                                                    , ezfb_get_g_from_cmap_index(fb, c)
                                                                                                    , ezfb_get_b_from_cmap_index(fb, c)));
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case 24: {
                                     int    local_x;
                                     u_char r, g, b;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width * 3; local_x += 3)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&b, sizeof(char));
                                         bytes_read += read(bmp->fd, (void*)&g, sizeof(char));
                                         bytes_read += read(bmp->fd, (void*)&r, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x / 3, i, ezfb_get_4bit_from_rgb(r, g, b));
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                     } // END switch(bmp->bpp)
                     break;
            // ---------------------------------------------------------------
            default:
            case  8: switch(bmp->bpp)
                     {
                        case  1: {
                                     int     local_x;
                                     u_char  c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x += 8)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x    , i, (c >> 7) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 1, i, (c >> 6) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 2, i, (c >> 5) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 3, i, (c >> 4) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 4, i, (c >> 3) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 5, i, (c >> 2) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 6, i, (c >> 1) & 1);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 7, i, (c     ) & 1);
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case  4: {
                                     int     local_x;
                                     u_char  c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x += 2)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x    , i, (c & 0xF0) >> 4);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 1, i,  c & 0x0F      );
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case  8: lseek(bmp->fd, junk_before, SEEK_CUR);
                                 bytes_read += read(bmp->fd, (void*)&(fb->screen_uchars[i * fb->Var.xres_virtual + fb_area->x1]), bmp_area->width);
                                 lseek(bmp->fd, junk_after, SEEK_CUR);
                                 break;
                        // ---------------------------------------------------
                        case 24: {
                                     int    local_x;
                                     u_char r, g, b;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width * 3; local_x += 3)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&b, sizeof(char));
                                         bytes_read += read(bmp->fd, (void*)&g, sizeof(char));
                                         bytes_read += read(bmp->fd, (void*)&r, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x / 3, i, (r & 0xe0) | ((g & 0xe0) >> 3) | ((b & 0xc0) >> 6));
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                     } // END switch(bmp->bpp)
                     break;
            // ---------------------------------------------------------------
            case 15:
            case 16:
            case 24:
            case 32: switch(bmp->bpp)
                     {
                        case  1: {
                                     int     local_x;
                                     u_int local_palette[] = {   ezfb_make_rgb(fb,   ezfb_get_r_from_cmap_index(fb, 0)
                                                                              , ezfb_get_g_from_cmap_index(fb, 0)
                                                                              , ezfb_get_b_from_cmap_index(fb, 0))
                                                               , ezfb_make_rgb(fb,   ezfb_get_r_from_cmap_index(fb, 1)
                                                                              , ezfb_get_g_from_cmap_index(fb, 1)
                                                                              , ezfb_get_b_from_cmap_index(fb, 1))
                                                             };
                                     u_char  c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x += 8)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x    , i, local_palette[(c >> 7) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 1, i, local_palette[(c >> 6) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 2, i, local_palette[(c >> 5) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 3, i, local_palette[(c >> 4) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 4, i, local_palette[(c >> 3) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 5, i, local_palette[(c >> 2) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 6, i, local_palette[(c >> 1) & 1]);
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 7, i, local_palette[(c     ) & 1]);
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case  4: {
                                     int    local_x;
                                     u_char c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x += 2)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         // ----------------------------------
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x,
                                                       i,
                                                       ezfb_make_rgb( fb,   ezfb_get_r_from_cmap_index(fb, ((c & 0xf0) >> 4))
                                                                     , ezfb_get_g_from_cmap_index(fb, ((c & 0xf0) >> 4))
                                                                     , ezfb_get_b_from_cmap_index(fb, ((c & 0xf0) >> 4)))
                                                  );
                                         // ----------------------------------
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x + 1,
                                                       i,
                                                       ezfb_make_rgb( fb,   ezfb_get_r_from_cmap_index(fb, (c & 0x0f))
                                                                     , ezfb_get_g_from_cmap_index(fb, (c & 0x0f))
                                                                     , ezfb_get_b_from_cmap_index(fb, (c & 0x0f)))
                                                  );
                                         // ----------------------------------
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case  8: {
                                     int    local_x;
                                     u_char c;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x++)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&c, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x,
                                                       i,
                                                       ezfb_make_rgb( fb,   ezfb_get_r_from_cmap_index(fb, c)
                                                                     , ezfb_get_g_from_cmap_index(fb, c)
                                                                     , ezfb_get_b_from_cmap_index(fb, c))
                                                  );
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                        case 24: {
                                     int    local_x;
                                     u_char r, g, b;
                                     lseek(bmp->fd, junk_before, SEEK_CUR);
                                     for(local_x = 0; local_x < bmp_area->width; local_x++)
                                     {
                                         bytes_read += read(bmp->fd, (void*)&b, sizeof(char));
                                         bytes_read += read(bmp->fd, (void*)&g, sizeof(char));
                                         bytes_read += read(bmp->fd, (void*)&r, sizeof(char));
                                         ezfb_put_pixel(fb, fb_area->x1 + local_x, i, ezfb_make_rgb(fb, r, g, b));
                                     }
                                     lseek(bmp->fd, junk_after, SEEK_CUR);
                                 }
                                 break;
                        // ---------------------------------------------------
                     } // END switch(bmp->bpp)
                     break;
            // ---------------------------------------------------------------
        } // END switch(fb->Var.bits_per_pixel)
    } // END for(i = (fb_area->y1 + bmp_area->height); i > fb_area->y1; i--)
    if(fb->Var.bits_per_pixel < bmp->bpp)
        ezfb_set_cmap_reduction(fb);
    ret = (bytes_read == bmp->image_size);
#if EZFB_API_MESSAGES
    printf("\n****** bytes_read = %d ******\n", bytes_read);fflush(stdout);
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_file_to_ezfb(struct ezfb* fb, char* file_name)
{
    struct ezfb_rectangle fb_area  = { -1, -1, -1, -1 , 0, 0 };
    struct ezfb_rectangle bmp_area = { -1, -1, -1, -1 , 0, 0 };
EZFB_FUNCTION_CALL_0
    ret = bmp_file_area_to_ezfb_at(fb, file_name, &fb_area, &bmp_area);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_file_to_ezfb_at(struct ezfb* fb, char* file_name, struct ezfb_rectangle* fb_area)
{
    struct ezfb_rectangle bmp_area = { -1, -1, -1, -1 , 0, 0 };
EZFB_FUNCTION_CALL_0
    ret = bmp_file_area_to_ezfb_at(fb, file_name, fb_area, &bmp_area);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_file_area_to_ezfb_at(struct ezfb* fb, char* file_name, struct ezfb_rectangle* fb_area, struct ezfb_rectangle* bmp_area)
{
    struct ezfb_bitmap  bmp;
EZFB_FUNCTION_CALL_0
    if(!fb->exists)
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: FB NOT INITIALIZED");fflush(stderr);
#endif
    }
    else if(bmp_open_for_read(&bmp, file_name)) // open the bitmap file
    {
        ret = bmp_read_file_header(&bmp);
#ifdef EZFB_API_MESSAGES
        bmp_dump_header(&bmp, file_name, "OPENING ....");
#endif
        ret &= bmp_file_palette_to_ezfb_(fb, &bmp);
        ret &= bmp_file_image_to_ezfb(fb, &bmp, fb_area, bmp_area);
        close(bmp.fd);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int bmp_file_to_ezfb_center(struct ezfb* fb, char* file_name)
{
    struct ezfb_bitmap       bmp;
    struct ezfb_rectangle    fb_area  = { -1, -1, -1, -1 , 0, 0 };
    struct ezfb_rectangle    bmp_area = { -1, -1, -1, -1 , 0, 0 };
EZFB_FUNCTION_CALL_0
    if(!fb->exists)
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: FB NOT INITIALIZED");fflush(stderr);
#endif
    }
    else if(bmp_open_for_read(&bmp, file_name)) // open the bitmap file
    {
        ret = bmp_read_file_header(&bmp);
#ifdef EZFB_API_MESSAGES
        bmp_dump_header(&bmp, file_name, "OPENING ....");
#endif
        ret &= bmp_file_palette_to_ezfb_(fb, &bmp);
        if(bmp.xres < fb->Var.xres)
            fb_area.x1 = (fb->max_x - bmp.xres) / 2;
        if(bmp.yres < fb->Var.yres)
            fb_area.y1 = (fb->max_y - bmp.yres) / 2;
#ifdef EZFB_API_MESSAGES
        ezfb_dump_rectangle(&fb_area, "fb_area");
#endif
        ret &= bmp_file_image_to_ezfb(fb, &bmp, &fb_area, &bmp_area);
            close(bmp.fd);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################

