//############################################################################
//
// EZFB ~ Linux Frame Buffer API ~
//
// by James Lehman
// james@akrobiz.com
//
// began: Feb. 2001
//
// EZFB is intended to give C programmers an easy-to-use library of functions
// for drawing points, lines and strings of characters into the Linux frame
// buffer. It also provides a means of displaying bitmap images on any portion
// of the screen and capturing the contents of any portion of the screen to
// a standard bitmap graphics file. All of this is embodied in a set of
// functions that are polymorphic with regard to the characteristics of the
// frame buffer and the bitmap files used. In other words, it makes no
// difference what the screen height or width is or what the color depth is,
// all of the function calls are the same and will have appropriate and
// predictable effects.
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#ifndef __bmp_FUNCTIONS__
#define __bmp_FUNCTIONS__

//############################################################################

#include "ezfb_types.h"

//############################################################################
//  UTILITY
int         ezfb_dump_rectangle       (struct ezfb_rectangle* area, char* message);
int         bmp_dump_header           (struct ezfb_bitmap* bmp, char* file_name, char* message);
//----------------------------------------------------------------------------
//  BITMAP IN MEMORY
int         bmp_create                (struct ezfb_bitmap* bmp, u_int x, u_int y, u_short bpp);
int         bmp_clone_ezfb_virtual    (struct ezfb* fb,         struct ezfb_bitmap* bmp, int copy_image);
int         bmp_clone_ezfb_screen     (struct ezfb* fb,         struct ezfb_bitmap* bmp, int copy_image);
int         bmp_put_pixel             (struct ezfb_bitmap* bmp, u_int x, u_int y, u_int c);
int         bmp_get_pixel             (struct ezfb_bitmap* bmp, u_int x, u_int y);
long double bmp_put_line              (struct ezfb_bitmap* bmp, int x1, int y1, int x2, int y2, u_int c);
long double bmp_put_ray               (struct ezfb_bitmap* bmp, uint x, uint y, long double r, long double t, uint c);
void        bmp_put_rectangle         (struct ezfb_bitmap* bmp, int x1, int y1, int x2, int y2, u_int c, int filled);
int         bmp_put_byte_pattern      (struct ezfb_bitmap* bmp, char   byte, int x, int y, int fc, int bc, int dir, int size);
int         bmp_put_ascii             (struct ezfb_bitmap* bmp, char  ascii, int x, int y, int fc, int bc, int dir, int size);
int         bmp_put_string            (struct ezfb_bitmap* bmp, char*   str, int x, int y, int fc, int bc, int dir, int size);
int         bmp_printf                (struct ezfb_bitmap* bmp,              int x, int y, int fc, int bc, int dir, int size, const char* text_format, ...);
int         bmp_clear                 (struct ezfb_bitmap* bmp, u_char c);
int         bmp_free                  (struct ezfb_bitmap* bmp);
//----------------------------------------------------------------------------
//  BITMAP MEMORY AND FILE
int         bmp_from_file             (struct ezfb_bitmap* bmp, char* file_name);
int         bmp_save_as               (struct ezfb_bitmap* bmp, char* file_name);
//----------------------------------------------------------------------------
//  FRAME BUFFER OUTPUT AS BMP FILE
int         bmp_save_ezfb_as          (struct ezfb* fb, char* file_name);
int         bmp_save_ezfb_area_as     (struct ezfb* fb, char* file_name, struct ezfb_rectangle* fb_area);
//----------------------------------------------------------------------------
//  FRAME BUFFER AND BMP IN MEMORY
int         bmp_to_ezfb               (struct ezfb* fb, struct ezfb_bitmap* bmp);
int         bmp_from_ezfb             (struct ezfb* fb, struct ezfb_bitmap* bmp);
int         bmp_palette_to_ezfb       (struct ezfb* fb, struct ezfb_bitmap* bmp);
int         bmp_palette_from_ezfb     (struct ezfb* fb, struct ezfb_bitmap* bmp);
//----------------------------------------------------------------------------
//  BMP FILE AS INPUT TO FRAME BUFFER
int         bmp_file_palette_to_ezfb  (struct ezfb* fb, char* file_name);
int         bmp_file_to_ezfb          (struct ezfb* fb, char* file_name);
int         bmp_file_to_ezfb_at       (struct ezfb* fb, char* file_name, struct ezfb_rectangle* fb_area);
int         bmp_file_area_to_ezfb_at  (struct ezfb* fb, char* file_name, struct ezfb_rectangle* fb_area, struct ezfb_rectangle* bmp_area);
int         bmp_file_to_ezfb_center   (struct ezfb* fb, char* file_name);

//############################################################################
#endif

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################

