//############################################################################
//
// EZFB ~ Linux Frame Buffer API ~
//
// by James Lehman
// james@akrobiz.com
//
// began: Feb. 2001
//
// EZFB is intended to give C programmers an easy-to-use library of functions
// for drawing points, lines and strings of characters into the Linux frame
// buffer. It also provides a means of displaying bitmap images on any portion
// of the screen and capturing the contents of any portion of the screen to
// a standard bitmap graphics file. All of this is embodied in a set of
// functions that are polymorphic with regard to the characteristics of the
// frame buffer and the bitmap files used. In other words, it makes no
// difference what the screen height or width is or what the color depth is,
// all of the function calls are the same and will have appropriate and
// predictable effects.
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ezfb_device.h"
#include "ezfb_pixels.h"
#include "ezfb_colors.h"

//############################################################################
int ezfb_get_cmap(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(0 > ioctl(fb->fd, FBIOGETCMAP, &fb->colors.fbcmap))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: FBIOGETCMAP FAILED");
#endif
        if(fb->Var.bits_per_pixel <= 8)
            ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(0 > ioctl(fb->fd, FBIOPUTCMAP, &fb->colors.fbcmap))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: FBIOPUTCMAP FAILED");
#endif
        if(fb->Var.bits_per_pixel <= 8)
            ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_get_color_factors(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    fb->colors.shortest_color = 17; // one higher than possible
    if(fb->colors.shortest_color > fb->Var.red.length)
        fb->colors.shortest_color = fb->Var.red.length;
    if(fb->colors.shortest_color > fb->Var.green.length)
        fb->colors.shortest_color = fb->Var.green.length;
    if(fb->colors.shortest_color > fb->Var.blue.length)
        fb->colors.shortest_color = fb->Var.blue.length;
    // masks a u_char msb left.
    fb->colors.bit_mask_red   = ((1 << fb->Var.red.length  ) - 1) << (EZFB_BITS_IN_CHAR - fb->Var.red.length  );
    fb->colors.bit_mask_green = ((1 << fb->Var.green.length) - 1) << (EZFB_BITS_IN_CHAR - fb->Var.green.length);
    fb->colors.bit_mask_blue  = ((1 << fb->Var.blue.length ) - 1) << (EZFB_BITS_IN_CHAR - fb->Var.blue.length );
    fb->colors.offset_plus_length_red   = fb->Var.red.offset   + fb->Var.red.length  ;
    fb->colors.offset_plus_length_green = fb->Var.green.offset + fb->Var.green.length;
    fb->colors.offset_plus_length_blue  = fb->Var.blue.offset  + fb->Var.blue.length ;
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init_cmap(struct ezfb* fb, int save_old)
{
EZFB_FUNCTION_CALL_1
    if(!fb->colors.exists)
    {
        fb->colors.fbcmap.start  = 0   ;
        fb->colors.fbcmap.transp = NULL;
        fb->colors.fbcmap.len    = (fb->Var.bits_per_pixel <= 8)
                                       ? (0x01 << fb->Var.bits_per_pixel)
                                       : 0;
        if(save_old)
        {
            fb->colors.fbcmap.red    = fb->colors.red_was   ; // connect pointers to user space
            fb->colors.fbcmap.green  = fb->colors.green_was ;
            fb->colors.fbcmap.blue   = fb->colors.blue_was  ;
            ret &= ezfb_get_cmap          (fb);                // fill user space with colors
            if(ret)
                 fb->colors.old_was_saved = 1;
            fb->colors.fbcmap.red    = fb->colors.red       ; // connect pointers to user space
            fb->colors.fbcmap.green  = fb->colors.green     ; // the ones we're really going to use
            fb->colors.fbcmap.blue   = fb->colors.blue      ;
            if(ret)
                 fb->colors.exists = 1;
            ret &= ezfb_get_color_factors (fb);
            ret &= ezfb_clear_cmap        (fb);
            ret &= ezfb_set_cmap_utility  (fb);
        }
        else
        {
            fb->colors.fbcmap.red    = fb->colors.red       ;
            fb->colors.fbcmap.green  = fb->colors.green     ;
            fb->colors.fbcmap.blue   = fb->colors.blue      ;
            ret &= ezfb_get_cmap          (fb);
            ret &= ezfb_get_color_factors (fb);
            if(ret)
            {
                 fb->colors.exists        = 1;
                 fb->colors.old_was_saved = 0;
            }
        }
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: ezfb_init_cmap(...) CALLED MORE THAN ONCE!");
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_clear_cmap(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < 256; i++)
        fb->colors.red[i] = fb->colors.green[i] = fb->colors.blue[i] = 0;
    ret = ezfb_set_cmap(fb);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_release_cmap(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(fb->colors.exists)
    {
        if(fb->colors.old_was_saved)
        {
            fb->colors.fbcmap.red    = fb->colors.red_was   ;
            fb->colors.fbcmap.green  = fb->colors.green_was ;
            fb->colors.fbcmap.blue   = fb->colors.blue_was  ;
            ret &= ezfb_set_cmap(fb); // put back the old color map
        }
        fb->colors.exists = 0;
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "\nezfb ERROR: (ezfb_release_cmap) FB CMAP NOT INITIALIZED");
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_find_black_and_white(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    switch(fb->Var.bits_per_pixel)
    {
        case  1:  fb->colors.black = 0;
                  fb->colors.white = 1;
                  break;
        //--------------------------------------------------------------------
        case  2:
        case  4:
        case  8: {
                      u_int   i;
                      fb->colors.black = fb->colors.white = 0;
                      for(i = 0; i < fb->colors.fbcmap.len; i++)
                      {
                          if(   (    (u_int)fb->colors.red   [fb->colors.white]
                                   + (u_int)fb->colors.green [fb->colors.white]
                                   + (u_int)fb->colors.blue  [fb->colors.white]
                                )
                              < (    (u_int)fb->colors.red   [i]
                                   + (u_int)fb->colors.green [i]
                                   + (u_int)fb->colors.blue  [i]
                                )
                            )
                              fb->colors.white = i;
                            //------------------------------------------------

                          if(   (    (u_int)fb->colors.red   [fb->colors.black]
                                   + (u_int)fb->colors.green [fb->colors.black]
                                   + (u_int)fb->colors.blue  [fb->colors.black]
                                )
                              > (    (u_int)fb->colors.red   [i]
                                   + (u_int)fb->colors.green [i]
                                   + (u_int)fb->colors.blue  [i]
                                )
                            )
                              fb->colors.black = i;
                            //------------------------------------------------
                      }
                  }
                  break;
        //--------------------------------------------------------------------
        case 15:
        case 16:
        case 24:
        case 32:
        default:  fb->colors.black = 0x00000000;
                  fb->colors.white = 0x00ffffff;
                  break;
    } // end switch(fb->Var.bits_per_pixel)
EZFB_FUNCTION_RETURN(fb->colors.exists)
}

//############################################################################
u_int ezfb_make_rgb(struct ezfb* fb, u_short r, u_short g, u_short b)
{
    switch(fb->Var.bits_per_pixel)
    {
        default:
        case  1:
        case  2: return ezfb_get_1bit_from_rgb(r, g, b);
        //--------------------------------------------------------------------
        case  4: return ezfb_get_4bit_from_rgb(r, g, b);
        //--------------------------------------------------------------------
        case  8: return (    (r & 0xe0)
                           | ((g & 0xe0) >> 3)
                           | ((b & 0xc0) >> 6)
                         );  // based on ezfb_set_cmap_reduction();
        //--------------------------------------------------------------------
        case 15:
        case 16: return ((   ((r & fb->colors.bit_mask_red  ) << fb->colors.offset_plus_length_red  )
                            | ((g & fb->colors.bit_mask_green) << fb->colors.offset_plus_length_green)
                            | ((b & fb->colors.bit_mask_blue ) << fb->colors.offset_plus_length_blue )  ) >> EZFB_BITS_IN_CHAR);
        //--------------------------------------------------------------------
        case 24:
        case 32: return   ((r & fb->colors.bit_mask_red  ) << fb->Var.red.offset  )
                         | ((g & fb->colors.bit_mask_green) << fb->Var.green.offset)
                         | ((b & fb->colors.bit_mask_blue ) << fb->Var.blue.offset );
    }
}

//############################################################################
u_short ezfb_get_r_from_rgb(struct ezfb* fb, u_int rgb)
{
    switch(fb->Var.bits_per_pixel)
    {
        default:
        case  1:
        case  2:
        case  4: return 0;
        //--------------------------------------------------------------------
        case  8: return (int)(255 * (rgb & 0xe0) / 224.0); // based on ezfb_set_cmap_reduction();
        //--------------------------------------------------------------------
        case 15:     // returns a scaled char ie: fb->Var.red.length bits to 8 bits
        case 16: {   // by adding the top msb's to the otherwise zeros at the low end
                      u_char c = ((((rgb) << EZFB_BITS_IN_CHAR) >> fb->colors.offset_plus_length_red) & fb->colors.bit_mask_red);
                      return c |= ((c & ~((1 << fb->Var.red.length) - 1)) >> fb->Var.red.length);
                  }   // off is off and full on is 255
        //--------------------------------------------------------------------
        case 24:
        case 32: return (rgb >> fb->Var.red.offset) & fb->colors.bit_mask_red;
    }
}

//############################################################################
u_short ezfb_get_g_from_rgb(struct ezfb* fb, u_int rgb)
{
    switch(fb->Var.bits_per_pixel)
    {
        default:
        case  1:
        case  2:
        case  4: return 0;
        //--------------------------------------------------------------------
        case  8: return (int)(255 * ((rgb & 0x1c) << 3) / 224.0); // based on ezfb_set_cmap_reduction();
        //--------------------------------------------------------------------
        case 15:
        case 16: {
                      u_char c = ((((rgb) << EZFB_BITS_IN_CHAR) >> fb->colors.offset_plus_length_green) & fb->colors.bit_mask_green);
                      return c |= ((c & ~((1 << fb->Var.green.length) - 1)) >> fb->Var.green.length);
                  }
        //--------------------------------------------------------------------
        case 24:
        case 32: return (rgb >> fb->Var.green.offset) & fb->colors.bit_mask_green;
    }
}

//############################################################################
u_short ezfb_get_b_from_rgb(struct ezfb* fb, u_int rgb)
{
    switch(fb->Var.bits_per_pixel)
    {
        default:
        case  1:
        case  2:
        case  4: return 0;
        //--------------------------------------------------------------------
        case  8: return (int)(255 * ((rgb & 0x03) << 6) / 192.0); // based on ezfb_set_cmap_reduction();
        //--------------------------------------------------------------------
        case 15:
        case 16: {
                      u_char c = ((((rgb) << EZFB_BITS_IN_CHAR) >> fb->colors.offset_plus_length_blue) & fb->colors.bit_mask_blue);
                      return c |= ((c & ~((1 << fb->Var.blue.length) - 1)) >> fb->Var.blue.length);
                  }
        //--------------------------------------------------------------------
        case 24:
        case 32: return (rgb >> fb->Var.blue.offset) & fb->colors.bit_mask_blue;
    }
}

//############################################################################
u_short ezfb_get_1bit_from_rgb(u_short r, u_short g, u_short b)
{
    if(r + g + b >= 384)
        return 1;
    return 0;
}
//############################################################################
u_short ezfb_get_4bit_from_rgb(u_short r, u_short g, u_short b)
{
/*
 0 0 0 0
 1 0 0 2
 2 0 2 0
 3 0 2 2
 4 2 0 0
 5 2 0 2
 6 2 2 0
 7 2 2 2
 8 1 1 1
 9 0 0 3
10 0 3 0
11 0 3 3
12 3 0 0
13 3 0 3
14 3 3 0
15 3 3 3
*/
    switch(r / 0x40)
    {
        case 0: // no red
                 switch(g / 0x40)
                 {
                     case 0: // no green
                     case 1: // 1/4 green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return  0; // 1/4 blue
                                  case 2: return  1; // 1/2 blue
                                  case 3: return  9; // all blue
                             }
                     case 2: // 1/2 green
                             switch(b / 0x40)
                             {
                                  case 0: return  2; //  no blue
                                  case 1: return  2; // 1/4 blue
                                  case 2: return  3; // 1/2 blue
                                  case 3: return 11; // all blue
                             }
                     case 3: // all green
                             switch(b / 0x40)
                             {
                                  case 0: return 10; //  no blue
                                  case 1: return 10; // 1/4 blue
                                  case 2: return 11; // 1/2 blue
                                  case 3: return 11; // all blue
                             }
                 }
        case 1: // 1/4 red
                 switch(g / 0x40)
                 {
                     case 0: // no green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return  0; // 1/4 blue
                                  case 2: return  1; // 1/2 blue
                                  case 3: return  9; // all blue
                             }
                     case 1: // 1/4 green
                             switch(b / 0x40)
                             {
                                  case 0: return  0; //  no blue
                                  case 1: return  8; // 1/4 blue
                                  case 2: return  1; // 1/2 blue
                                  case 3: return  9; // all blue
                             }
                     case 2: // 1/2 green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return  2; // 1/4 blue
                                  case 2: return  3; // 1/2 blue
                                  case 3: return 11; // all blue
                             }
                     case 3: // all green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return 10; // 1/4 blue
                                  case 2:            // 1/2 blue
                                  case 3: return 11; // all blue
                             }
                 }
        case 2: // 1/2 red
                 switch(g / 0x40)
                 {
                     case 0: // no green
                     case 1: // 1/4 green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return  4; // 1/4 blue
                                  case 2: return  5; // 1/2 blue
                                  case 3: return 13; // all blue
                             }
                     case 2: // 1/2 green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return  6; // 1/4 blue
                                  case 2: return  7; // 1/2 blue
                                  case 3: return 15; // all blue
                             }
                     case 3: // all green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return 14; // 1/4 blue
                                  case 2:            // 1/2 blue
                                  case 3: return 15; // all blue
                             }
                 }
        case 3: // all read
                 switch(g / 0x40)
                 {
                     case 0: // no green
                     case 1: // 1/4 green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return 12; // 1/4 blue
                                  case 2:            // 1/2 blue
                                  case 3: return 13; // all blue
                             }
                     case 2: // 1/2 green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return 14; // 1/4 blue
                                  case 2:            // 1/2 blue
                                  case 3: return 15; // all blue
                             }
                     case 3: // all green
                             switch(b / 0x40)
                             {
                                  case 0:            //  no blue
                                  case 1: return 14; // 1/4 blue
                                  case 2:            // 1/2 blue
                                  case 3: return 15; // all blue
                             }
                 }
    }
    return 0;
}

//############################################################################
u_int ezfb_rgb_hue(struct ezfb* fb, u_short hue)
{
    u_short r = 0, g = 0, b = 0;
    u_char  shift;
    hue %= 1530;
    shift = hue % 255;
    if(hue < 255)
    {
        r = 255;
        g = shift;
    }
    else if(hue < 510)
    {
        r = 256 - shift;
        g = 255;
    }
    else if(hue < 765)
    {
        g = 255;
        b = shift;
    }
    else if(hue < 1020)
    {
        g = 256 - shift;
        b = 255;
    }
    else if(hue < 1275)
    {
        r = shift;
        b = 255;
    }
    else
    {
        r = 255;
        b = 256 - shift;
    }
    return ezfb_make_rgb(fb, r, g, b);
}

//############################################################################
u_int ezfb_rgb_tint(struct ezfb* fb, u_short hue, u_char tint)
{
    u_short r = 0,
            g = 0,
            b = 0;
    u_char  shift;
    hue %= 1530;
    shift = hue % 255;
    if(hue < 255)
    {
        r = 255;
        g = (u_short)(shift + (tint * (255 - shift) / 255.0));
        g = (g > 255) ? (255) : (g);
        b = tint;
    }
    else if(hue < 510)
    {
        r = (u_short)((256 - shift) + (tint * shift / 255.0));
        r = (r > 255) ? (255) : (r);
        g = 255;
        b = tint;
    }
    else if(hue < 765)
    {
        r = tint;
        g = 255;
        b = (u_short)(shift + (tint * (255 - shift) / 255.0));
        b = (b > 255) ? (255) : (b);
    }
    else if(hue < 1020)
    {
        r = tint;
        g = (u_short)((256 - shift) + (tint * shift / 255.0));
        g = (g > 255) ? (255) : (g);
        b = 255;
    }
    else if(hue < 1275)
    {
        r = (u_short)(shift + (tint * (255 - shift) / 255.0));
        r = (r > 255) ? (255) : (r);
        g = tint;
        b = 255;
    }
    else
    {
        r = 255;
        g = tint;
        b = (u_short)((256 - shift) + (tint * shift / 255.0));
        b = (b > 255) ? (255) : (b);
    }
    return ezfb_make_rgb(fb, r, g, b);
}

//############################################################################
u_int ezfb_rgb_shade(struct ezfb* fb, u_short hue, u_char shade)
{
    u_short r = 0,
            g = 0,
            b = 0;
    u_char  shift;
    float   factor = (255 - shade) / 255.0;
    hue %= 1530;
    shift = hue % 255;
    if(hue < 255)
    {
        r = (u_short)(255   * factor);
        g = (u_short)(shift * factor);
    }
    else if(hue < 510)
    {
        r = (u_short)((255 - shift) * factor);
        g = (u_short) (255          * factor);
    }
    else if(hue < 765)
    {
        g = (u_short)(255   * factor);
        b = (u_short)(shift * factor);
    }
    else if(hue < 1020)
    {
        g = (u_short)((255 - shift) * factor);
        b = (u_short)( 255          * factor);
    }
    else if(hue < 1275)
    {
        r = (u_short)(shift * factor);
        b = (u_short)(255   * factor);
    }
    else
    {
        r = (u_short)( 255          * factor);
        b = (u_short)((255 - shift) * factor);
    }
    return ezfb_make_rgb(fb, r, g, b);
}

//############################################################################
int ezfb_set_cmap_to_was(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_0
    if(fb->colors.exists)
    {
        memcpy((void*)fb->colors.red   , (const void*)fb->colors.red_was   , 256);
        memcpy((void*)fb->colors.green , (const void*)fb->colors.green_was , 256);
        memcpy((void*)fb->colors.blue  , (const void*)fb->colors.blue_was  , 256);
        ret = ezfb_set_cmap(fb);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_grays(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < (1 << fb->Var.bits_per_pixel); i++)
        ezfb_set_cmap_index(fb, i, (i % (1 << fb->colors.shortest_color)) << (EZFB_BITS_IN_SHORT - fb->colors.shortest_color),
                                     (i % (1 << fb->colors.shortest_color)) << (EZFB_BITS_IN_SHORT - fb->colors.shortest_color),
                                     (i % (1 << fb->colors.shortest_color)) << (EZFB_BITS_IN_SHORT - fb->colors.shortest_color)
                          );
    if(fb->colors.exists)
    {
        ret = ezfb_set_cmap(fb);
        fb->colors.black = 0;
        fb->colors.white = (1 << fb->colors.shortest_color) - 1;
    }

EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_reds(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < (1 << fb->Var.bits_per_pixel); i++)
        ezfb_set_cmap_index(fb, i, (i % (1 << fb->Var.red.length)) << (EZFB_BITS_IN_SHORT - fb->Var.red.length), 0, 0);
    if(fb->colors.exists)
    {
        ret = ezfb_set_cmap(fb);
        fb->colors.black = 0;
        fb->colors.white = (1 << fb->Var.red.length) - 1;
    }

EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_greens(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < (1 << fb->Var.bits_per_pixel); i++)
        ezfb_set_cmap_index(fb, i, 0, (i % (1 << fb->Var.green.length)) << (EZFB_BITS_IN_SHORT - fb->Var.green.length), 0);
    if(fb->colors.exists)
    {
        ret = ezfb_set_cmap(fb);
        fb->colors.black = 0;
        fb->colors.white = (1 << fb->Var.green.length) - 1;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_blues(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < (1 << fb->Var.bits_per_pixel); i++)
        ezfb_set_cmap_index(fb, i, 0, 0, (i % (1 << fb->Var.blue.length)) << (EZFB_BITS_IN_SHORT - fb->Var.blue.length));
    if(fb->colors.exists)
    {
        ret = ezfb_set_cmap(fb);
        fb->colors.black = 0;
        fb->colors.white = (1 << fb->Var.blue.length) - 1;
    }
EZFB_FUNCTION_RETURN(ret)
}


//############################################################################
int ezfb_set_cmap_hues(struct ezfb* fb)
{
    u_short r = 0, g = 0, b = 0;
    u_char  shift;
    int i, hue;
EZFB_FUNCTION_CALL_0
    for(i = 2; i < (1 << fb->Var.bits_per_pixel) - 1; i++)
    {
        hue = i * (1530 / ((1 << fb->Var.bits_per_pixel) - 3));
        shift = hue % 255;
        if(hue < 255)
        {
            r = 255;
            g = shift;
        }
        else if(hue < 510)
        {
            r = 256 - shift;
            g = 255;
        }
        else if(hue < 765)
        {
            g = 255;
            b = shift;
        }
        else if(hue < 1020)
        {
            g = 256 - shift;
            b = 255;
        }
        else if(hue < 1275)
        {
            r = shift;
            b = 255;
        }
        else
        {
            r = 255;
            b = 256 - shift;
        }
        ezfb_set_cmap_index(fb, i, r << (EZFB_BITS_IN_SHORT - fb->Var.red.length   ),
                                   g << (EZFB_BITS_IN_SHORT - fb->Var.green.length ),
                                   b << (EZFB_BITS_IN_SHORT - fb->Var.blue.length  ));
    }
    ezfb_set_cmap_index(fb,   0, 0xff << (EZFB_BITS_IN_SHORT - fb->Var.red.length   ),
                                 0xff << (EZFB_BITS_IN_SHORT - fb->Var.green.length ),
                                 0xff << (EZFB_BITS_IN_SHORT - fb->Var.blue.length  ));
    ezfb_set_cmap_index(fb,   1, 0x80 << (EZFB_BITS_IN_SHORT - fb->Var.red.length   ),
                                 0x80 << (EZFB_BITS_IN_SHORT - fb->Var.green.length ),
                                 0x80 << (EZFB_BITS_IN_SHORT - fb->Var.blue.length  ));
    ezfb_set_cmap_index(fb, 255, 0x00 << (EZFB_BITS_IN_SHORT - fb->Var.red.length   ),
                                 0x00 << (EZFB_BITS_IN_SHORT - fb->Var.green.length ),
                                 0x00 << (EZFB_BITS_IN_SHORT - fb->Var.blue.length  ));
    if(fb->colors.exists)
    {
        ret = ezfb_set_cmap(fb);
        fb->colors.black = 0;
        fb->colors.white = (1 << fb->Var.blue.length) - 1;
    }
EZFB_FUNCTION_RETURN(ret)
}


//############################################################################
int ezfb_set_cmap_utility(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_0
    if(1 == fb->Var.bits_per_pixel)
    {
        ezfb_set_cmap_index(fb,  0, 0x0000, 0x0000, 0x0000); // black
        ezfb_set_cmap_index(fb,  1, 0xffff, 0xffff, 0xffff); // white
        fb->colors.black = 0;
        fb->colors.white = 1;
    }
    else if(2 == fb->Var.bits_per_pixel)
    {
        ezfb_set_cmap_index(fb,  0, 0x0000, 0x0000, 0x0000); // black
        ezfb_set_cmap_index(fb,  1, 0xffff, 0x0000, 0x0000); // red
        ezfb_set_cmap_index(fb,  2, 0x0000, 0xffff, 0xffff); // cyan
        ezfb_set_cmap_index(fb,  3, 0xffff, 0xffff, 0xffff); // white
        fb->colors.black = 0;
        fb->colors.white = 3;
    }
    else if((4 == fb->Var.bits_per_pixel) || (8 == fb->Var.bits_per_pixel))
    {   // default console colors
        ezfb_set_cmap_index(fb,  0, 0x0000, 0x0000, 0x0000); // black
        ezfb_set_cmap_index(fb,  1, 0x0000, 0x0000, 0xa000);
        ezfb_set_cmap_index(fb,  2, 0x0000, 0xa000, 0x0000);
        ezfb_set_cmap_index(fb,  3, 0x0000, 0xa000, 0xa000);
        ezfb_set_cmap_index(fb,  4, 0xa000, 0x0000, 0x0000);
        ezfb_set_cmap_index(fb,  5, 0xa000, 0x0000, 0xa000);
        ezfb_set_cmap_index(fb,  6, 0xa000, 0xa000, 0x0000);
        ezfb_set_cmap_index(fb,  7, 0xa000, 0xa000, 0xa000);
        ezfb_set_cmap_index(fb,  8, 0x4000, 0x4000, 0x4000);
        ezfb_set_cmap_index(fb,  9, 0x4000, 0x4000, 0xf000);
        ezfb_set_cmap_index(fb, 10, 0x4000, 0xf000, 0x4000);
        ezfb_set_cmap_index(fb, 11, 0x4000, 0xf000, 0xf000);
        ezfb_set_cmap_index(fb, 12, 0xf000, 0x4000, 0x4000);
        ezfb_set_cmap_index(fb, 13, 0xf000, 0x4000, 0xf000);
        ezfb_set_cmap_index(fb, 14, 0xf000, 0xf000, 0x4000);
        ezfb_set_cmap_index(fb, 15, 0xf000, 0xf000, 0xf000);
        fb->colors.black = 0;
        fb->colors.white = 15;
    }
    if(8 == fb->Var.bits_per_pixel)
    {
        int i;
        for(i = 0; i < 16; i++)
        {
            ezfb_set_cmap_index(fb, i +  16, i * 0x1000, i * 0x1000, i * 0x1000); // grays
            ezfb_set_cmap_index(fb, i +  32, i * 0x1000,          0,          0); // reds
            ezfb_set_cmap_index(fb, i +  48,          0, i * 0x1000,          0); // greens
            ezfb_set_cmap_index(fb, i +  64,          0,          0, i * 0x1000); // blues
            ezfb_set_cmap_index(fb, i +  80, i * 0x1000, i * 0x1000,          0); // yellows
            ezfb_set_cmap_index(fb, i +  96, i * 0x1000,          0, i * 0x1000); // magentas
            ezfb_set_cmap_index(fb, i + 112,          0, i * 0x1000, i * 0x1000); // cyans
            ezfb_set_cmap_index(fb, i + 128, i * 0x1000, i * 0x0800,          0); // oranges
            ezfb_set_cmap_index(fb, i + 144, i * 0x1000,          0, i * 0x0800); // hot pinks
            ezfb_set_cmap_index(fb, i + 160, i * 0x0800,          0, i * 0x1000); // purples
            ezfb_set_cmap_index(fb, i + 176,          0, i * 0x1000, i * 0x0800); // mints
            ezfb_set_cmap_index(fb, i + 192,          0, i * 0x0800, i * 0x1000); // light blues
            ezfb_set_cmap_index(fb, i + 208, i * 0x1000, i * 0x0800, i * 0x0800); // salmons
            ezfb_set_cmap_index(fb, i + 224, i * 0x0800, i * 0x0800, i * 0x1000); // light indigos
        }
        ezfb_set_cmap_index(fb, 240, 0xc000, 0xa000, 0x4000); // tan
        ezfb_set_cmap_index(fb, 241, 0x8000, 0x4000, 0x8000); // light purple
        ezfb_set_cmap_index(fb, 242, 0x8000, 0x8000, 0xffff); // light blue
        ezfb_set_cmap_index(fb, 243, 0xffff, 0x8000, 0x8000); // pink
        ezfb_set_cmap_index(fb, 244, 0xffff, 0x0000, 0x8000); // hot pink
        ezfb_set_cmap_index(fb, 245, 0x8000, 0x0000, 0x8000); // purple
        ezfb_set_cmap_index(fb, 246, 0xffff, 0x8000, 0x0000); // orange
        ezfb_set_cmap_index(fb, 247, 0x8000, 0x8000, 0x8000); // midtone gray
        ezfb_set_cmap_index(fb, 248, 0xffff, 0xffff, 0x0000); // yellow
        ezfb_set_cmap_index(fb, 249, 0xffff, 0x0000, 0xffff); // magenta
        ezfb_set_cmap_index(fb, 250, 0x0000, 0xffff, 0xffff); // cyan
        ezfb_set_cmap_index(fb, 251, 0x0000, 0x0000, 0xffff); // blue
        ezfb_set_cmap_index(fb, 252, 0x0000, 0xffff, 0x0000); // green
        ezfb_set_cmap_index(fb, 253, 0xffff, 0x0000, 0x0000); // red
        ezfb_set_cmap_index(fb, 254, 0x0000, 0x0000, 0x0000); // black
        ezfb_set_cmap_index(fb, 255, 0xffff, 0xffff, 0xffff); // white
        fb->colors.black = 0;
        fb->colors.white = 255;
    }
    if(fb->colors.exists)
        ret = ezfb_set_cmap(fb);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_smooth(struct ezfb* fb)
{
    int   i;
    float pi   = 4 * atan(1),
          Rfac = ezfb_random_01() * (pi / (fb->colors.fbcmap.len / 4.0)),
          Roff = ezfb_random_01() * 2 * pi,
          Gfac = ezfb_random_01() * (pi / (fb->colors.fbcmap.len / 4.0)),
          Goff = ezfb_random_01() * 2 * pi,
          Bfac = ezfb_random_01() * (pi / (fb->colors.fbcmap.len / 4.0)),
          Boff = ezfb_random_01() * 2 * pi;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < (1 << fb->Var.bits_per_pixel); i++)
        ezfb_set_cmap_index(fb, i, (u_short)((sin(Roff + Rfac * i) + 1) * 0x7fff),
                                   (u_short)((sin(Goff + Gfac * i) + 1) * 0x7fff),
                                   (u_short)((sin(Boff + Bfac * i) + 1) * 0x7fff)    );
    if(fb->colors.exists)
    {
        ret =  ezfb_set_cmap(fb);
        ret &= ezfb_find_black_and_white(fb);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_random(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    for(i = 0; i < 256; i++)
        ezfb_set_cmap_index(fb, i, (u_short)((random() % 0x00010000)),
                                   (u_short)((random() % 0x00010000)),
                                   (u_short)((random() % 0x00010000)) );
    if(fb->colors.exists)
    {
        ret =  ezfb_set_cmap(fb);
        ret &= ezfb_find_black_and_white(fb);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_cmap_reduction(struct ezfb* fb)
{
    int i;
EZFB_FUNCTION_CALL_0
    switch(fb->Var.bits_per_pixel)
    {
        case  1: ezfb_set_cmap_index(fb, 0, 0x0000, 0x0000, 0x0000); // black
                 ezfb_set_cmap_index(fb, 1, 0xffff, 0xffff, 0xffff); // white
                 break;
        //--------------------------------------------------------------------
        case  2: ezfb_set_cmap_index(fb, 0, 0x0000, 0x0000, 0x0000); // black
                 ezfb_set_cmap_index(fb, 1, 0xffff, 0x0000, 0xffff); // magenta
                 ezfb_set_cmap_index(fb, 2, 0x0000, 0xffff, 0xffff); // cyan
                 ezfb_set_cmap_index(fb, 3, 0xffff, 0xffff, 0xffff); // white
                 break;
        //--------------------------------------------------------------------
        case  4: ezfb_set_cmap_index(fb,  0, 0x0000, 0x0000, 0x0000); // black
                 ezfb_set_cmap_index(fb,  1, 0x0000, 0x0000, 0x7f00);
                 ezfb_set_cmap_index(fb,  2, 0x0000, 0x7f00, 0x0000);
                 ezfb_set_cmap_index(fb,  3, 0x0000, 0x7f00, 0x7f00);
                 ezfb_set_cmap_index(fb,  4, 0x7f00, 0x0000, 0x0000);
                 ezfb_set_cmap_index(fb,  5, 0x7f00, 0x0000, 0x7f00);
                 ezfb_set_cmap_index(fb,  6, 0x7f00, 0x7f00, 0x0000);
                 ezfb_set_cmap_index(fb,  7, 0x7f00, 0x7f00, 0x7f00);
                 ezfb_set_cmap_index(fb,  8, 0x3f00, 0x3f00, 0x3f00);
                 ezfb_set_cmap_index(fb,  9, 0x0000, 0x0000, 0xff00);
                 ezfb_set_cmap_index(fb, 10, 0x0000, 0xff00, 0x0000);
                 ezfb_set_cmap_index(fb, 11, 0x0000, 0xff00, 0xff00);
                 ezfb_set_cmap_index(fb, 12, 0xff00, 0x0000, 0x0000);
                 ezfb_set_cmap_index(fb, 13, 0xff00, 0x0000, 0xff00);
                 ezfb_set_cmap_index(fb, 14, 0xff00, 0xff00, 0x0000);
                 ezfb_set_cmap_index(fb, 15, 0xff00, 0xff00, 0xff00);
                 break;
        //--------------------------------------------------------------------
        case  8:
        case 15:
        case 16:
        case 24:
        case 32: for(i = 0; i < 256; i++)
                       ezfb_set_cmap_index(fb, i, ((int)(255 *  (i & 0xe0)       / 224.0)) << EZFB_BITS_IN_CHAR
                                                , ((int)(255 * ((i & 0x1c) << 3) / 224.0)) << EZFB_BITS_IN_CHAR
                                                , ((int)(255 * ((i & 0x03) << 6) / 192.0)) << EZFB_BITS_IN_CHAR);
                  break;
        //--------------------------------------------------------------------
    }
    if(fb->colors.exists && (fb->Var.bits_per_pixel <= 8))
    {
        ret =  ezfb_set_cmap(fb);
        ret &= ezfb_find_black_and_white(fb);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
void ezfb_set_cmap_index(struct ezfb* fb, int i, u_short r, u_short g, u_short b)
{
    if((i >= 0) & (i < 256))
    {
        fb->colors.red   [i] = fb->Var.red.msb_right   ? (r >> (EZFB_BITS_IN_SHORT - fb->Var.red.length   )) : r;
        fb->colors.green [i] = fb->Var.green.msb_right ? (g >> (EZFB_BITS_IN_SHORT - fb->Var.green.length )) : g;
        fb->colors.blue  [i] = fb->Var.blue.msb_right  ? (b >> (EZFB_BITS_IN_SHORT - fb->Var.blue.length  )) : b;
    }
}

//############################################################################
void ezfb_get_cmap_index(struct ezfb* fb, int i, u_short* r, u_short* g, u_short* b)
{
    if((i >= 0) & (i < 256))
    {
        *r = fb->Var.red.msb_right   ? (fb->colors.red   [i] << (EZFB_BITS_IN_SHORT - fb->Var.red.length   )) : fb->colors.red   [i];
        *g = fb->Var.green.msb_right ? (fb->colors.green [i] << (EZFB_BITS_IN_SHORT - fb->Var.green.length )) : fb->colors.green [i];
        *b = fb->Var.blue.msb_right  ? (fb->colors.blue  [i] << (EZFB_BITS_IN_SHORT - fb->Var.blue.length  )) : fb->colors.blue  [i];
    }
}

//############################################################################
u_short ezfb_get_r_from_cmap_index(struct ezfb* fb, int i)
{
    if((i >= 0) & (i < 256))
        return (fb->Var.red.msb_right ? fb->colors.red[i] : fb->colors.red[i] >> EZFB_BITS_IN_CHAR) & fb->colors.bit_mask_red;
    return 0;
}

//############################################################################
u_short ezfb_get_g_from_cmap_index(struct ezfb* fb, int i)
{
    if((i >= 0) & (i < 256))
        return (fb->Var.green.msb_right ? fb->colors.green[i] : fb->colors.green[i] >> EZFB_BITS_IN_CHAR) & fb->colors.bit_mask_green;
    return 0;
}

//############################################################################
u_short ezfb_get_b_from_cmap_index(struct ezfb* fb, int i)
{
    if((i >= 0) & (i < 256))
        return (fb->Var.blue.msb_right ? fb->colors.blue[i] : fb->colors.blue[i] >> EZFB_BITS_IN_CHAR) & fb->colors.bit_mask_blue;
    return 0;
}

//############################################################################
int ezfb_cycle_cmap(struct ezfb* fb, u_char from_index, u_char to_index, int step)
{
    if(fb->colors.exists && step && (from_index != to_index))
    {
        u_char  i, j;
        u_short r, g, b;
        if(from_index > to_index)
        {
            i          = from_index ;
            from_index = to_index   ;
            to_index   = i          ;
        }
        if(step > 0)
            for(j = 0; j < step; j++)
            {
                 r = fb->colors.red   [from_index];
                 g = fb->colors.green [from_index];
                 b = fb->colors.blue  [from_index];
                 for(i = from_index; i < to_index; i++)
                 {
                     fb->colors.red   [i] = fb->colors.red   [i + 1];
                     fb->colors.green [i] = fb->colors.green [i + 1];
                     fb->colors.blue  [i] = fb->colors.blue  [i + 1];
                 }
                 fb->colors.red   [to_index] = r;
                 fb->colors.green [to_index] = g;
                 fb->colors.blue  [to_index] = b;
            }
        else
        {
            step = -step;
            for(j = 0; j < step; j++)
            {
                 r = fb->colors.red   [to_index];
                 g = fb->colors.green [to_index];
                 b = fb->colors.blue  [to_index];
                 for(i = to_index; i > from_index; i--)
                 {
                     fb->colors.red   [i] = fb->colors.red   [i - 1];
                     fb->colors.green [i] = fb->colors.green [i - 1];
                     fb->colors.blue  [i] = fb->colors.blue  [i - 1];
                 }
                 fb->colors.red   [from_index] = r;
                 fb->colors.green [from_index] = g;
                 fb->colors.blue  [from_index] = b;
            }
        }
        return ezfb_set_cmap(fb);
    }
    return 0;
}

//############################################################################
//int fb_cmap_color_parade()
//{
//
//}

//############################################################################
int ezfb_set_background(struct ezfb* fb, u_int rgb)
{
EZFB_NIT_PICKING_FUNCTION_CALL_1
    ret = ezfb_put_rectangle(fb, 0, 0, fb->max_x, fb->max_y, rgb, 1);
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_dump_colors(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
#if EZFB_API_MESSAGES
    printf("\n      EZFB COLORS:"                                                   );
    printf("\n      fb->colors.exists         = %u"  , fb->colors.exists            );
    printf("\n      fb->colors.white          = 0x%X", fb->colors.white             );
    printf("\n      fb->colors.black          = 0x%X", fb->colors.black             );
    printf("\n      fb->colors.bit_mask_red   = 0x%X", fb->colors.bit_mask_red      );
    printf("\n      fb->colors.bit_mask_green = 0x%X", fb->colors.bit_mask_green    );
    printf("\n      fb->colors.bit_mask_blue  = 0x%X", fb->colors.bit_mask_blue     );
    printf("\n      fb->colors.shortest_color = %u"  , fb->colors.shortest_color    );
    printf("\n      fb->colors.red            = 0x%X", (u_int) fb->colors.red       );
    printf("\n      fb->colors.green          = 0x%X", (u_int) fb->colors.green     );
    printf("\n      fb->colors.blue           = 0x%X", (u_int) fb->colors.blue      );
    printf("\n      fb->colors.red_was        = 0x%X", (u_int) fb->colors.red_was   );
    printf("\n      fb->colors.green_was      = 0x%X", (u_int) fb->colors.green_was );
    printf("\n      fb->colors.blue_was       = 0x%X", (u_int) fb->colors.blue_was  );
    printf("\n\n");
    fflush(stdout);
#endif
EZFB_FUNCTION_RETURN(fb->colors.exists)
};

//############################################################################
int ezfb_dump_cmap(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->colors.exists)
    {
#if EZFB_API_MESSAGES
        printf("\n      FBCMAP:");
        printf("\n      fbcmap.start  = %u"  , fb->colors.fbcmap.start        );
        printf("\n      fbcmap.len    = %u"  , fb->colors.fbcmap.len          );
        printf("\n      fbcmap.red    = 0x%X", (u_int)fb->colors.fbcmap.red   );
        printf("\n      fbcmap.green  = 0x%X", (u_int)fb->colors.fbcmap.green );
        printf("\n      fbcmap.blue   = 0x%X", (u_int)fb->colors.fbcmap.blue  );
        printf("\n      fbcmap.transp = 0x%X", (u_int)fb->colors.fbcmap.transp);
        printf("\n\n");
        fflush(stdout);
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "\nezfb ERROR: (ezfb_dump_cmap) FB CMAP NOT INITIALIZED");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->colors.exists)
}

//############################################################################
int ezfb_dump_cmap_values(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->colors.exists)
    {
#if EZFB_API_MESSAGES
        int i;
        printf("\nFB CMAP SINCE THIS APPLICATION CHANGED IT:");
        printf("\n[index]. [ red ]. [green]. [blue ]");
        for(i = 0; i < 256; i++)
            printf("\n   %3d . 0x%4X . 0x%4X . 0x%4X", i, fb->colors.red   [i],
                                                          fb->colors.green [i],
                                                          fb->colors.blue  [i]  );
        printf("\n\n");
        fflush(stdout);
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "\nezfb ERROR: (ezfb_dump_cmap_values) FB CMAP NOT INITIALIZED");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->colors.exists)
}

//############################################################################
int ezfb_dump_cmap_was_values(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->colors.exists)
    {
#if EZFB_API_MESSAGES
        int i;
        printf("\nFB CMAP BEFORE THIS APPLICATION CHANGED IT:");
        printf("\n[index]. [ red ]. [green]. [blue ]");
        for(i = 0; i < 256; i++)
            printf("\n   %3d . 0x%4X . 0x%4X . 0x%4X", i, fb->colors.red_was   [i],
                                                          fb->colors.green_was [i],
                                                          fb->colors.blue_was  [i]  );
        printf("\n\n");
        fflush(stdout);
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "\nezfb ERROR: (ezfb_dump_cmap_was_values) FB CMAP NOT INITIALIZED");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->colors.exists)
}

//############################################################################
int ezfb_binfile_cmap(struct ezfb* fb, char* file_name)
{
EZFB_FUNCTION_CALL_0
    if(fb->colors.exists)
    {
        int i, binfile_fd;
        char c;
        if(0 > (binfile_fd = open(file_name, (O_WRONLY | O_CREAT), (S_IRWXU | S_IRGRP | S_IROTH)))) // mode 744
        {
#if EZFB_API_ERRORS
            perror("\nezfb ERROR: OPENING OUTPUT FILE FAILED");
#endif
            ret = 0;
        }
        else
        {
            ret = 1;
            for(i = 0; i < (int)(fb->colors.fbcmap.len); i++)
            {
                 c = (char)ezfb_get_r_from_cmap_index(fb, i);
                 write(binfile_fd, (void*)(&c), 1);
                 c = (char)ezfb_get_g_from_cmap_index(fb, i);
                 write(binfile_fd, (void*)(&c), 1);
                 c = (char)ezfb_get_b_from_cmap_index(fb, i);
                 write(binfile_fd, (void*)(&c), 1);
                 c = 0;
                 write(binfile_fd, (void*)(&c), 1);
            }
            close(binfile_fd);
        }
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "\nezfb ERROR: (ezfb_binfile_cmap) FB CMAP NOT INITIALIZED");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################

