//############################################################################
//
// EZFB ~ Linux Frame Buffer API ~
//
// by James Lehman
// james@akrobiz.com
//
// began: Feb. 2001
//
// EZFB is intended to give C programmers an easy-to-use library of functions
// for drawing points, lines and strings of characters into the Linux frame
// buffer. It also provides a means of displaying bitmap images on any portion
// of the screen and capturing the contents of any portion of the screen to
// a standard bitmap graphics file. All of this is embodied in a set of
// functions that are polymorphic with regard to the characteristics of the
// frame buffer and the bitmap files used. In other words, it makes no
// difference what the screen height or width is or what the color depth is,
// all of the function calls are the same and will have appropriate and
// predictable effects.
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ezfb_tty.h"
#include "ezfb_colors.h"
#include "ezfb_device.h"

//############################################################################
int    ezfb_open_fb         (struct ezfb* fb, char* dev_name);
int    ezfb_get_fb_var      (struct ezfb* fb);
int    ezfb_set_fb_var      (struct ezfb* fb);
int    ezfb_get_fb_fix      (struct ezfb* fb);
int    ezfb_get_fb_mem      (struct ezfb* fb, int save_old);
int    ezfb_find_screen     (struct ezfb* fb);
int    ezfb_init_base       (struct ezfb* fb, char* dev_name, int save_old);
int    ezfb_copy_old_fb     (struct ezfb* fb);
void   ezfb_set_sig_traps   ();
void   ezfb_terminate       (int signum);

//############################################################################
struct ezfb* valid_ezfb[EZFB_MAX_FB_NUM + 1] = {NULL};
u_int        num_valid_ezfb             = 0;

//############################################################################
int ezfb_dev_exists(char* dev_name)
{
    int fd;
EZFB_FUNCTION_CALL_1
    if(0 > (fd = open(dev_name, O_RDWR)))
    {
#if EZFB_API_ERRORS
        perror("ezfb ERROR: FRAME BUFFER DOES NOT EXIST ");
        printf("(%s) FAILED", dev_name);
#endif
        ret = 0;
    }
    else
    {
#if EZFB_API_MESSAGES
        printf("FRAME BUFFER %s EXSISTS\n", dev_name);
        fflush(stdout);
#endif
        close(fd);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_num_exists(int num)
{
    char dev_name[16] = {0};
EZFB_FUNCTION_CALL_1
    strcpy(dev_name, EZFB_DEV_NAME);
    sprintf(&(dev_name[strlen(dev_name)]), "%d", num);
    ret = ezfb_dev_exists(dev_name);
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
u_int ezfb_name_to_num(char* dev_name) // returns last one or two digits as u_int
{
    if(isdigit(dev_name[strlen(dev_name) - 2]))
        return (u_int)atoi(&dev_name[strlen(dev_name) - 2]);
    return (u_int)atoi(&dev_name[strlen(dev_name) - 1]);
}

//############################################################################
int ezfb_open_fb(struct ezfb* fb, char* dev_name)
{
EZFB_FUNCTION_CALL_1
    if(0 > (fb->fd = open(dev_name, O_RDWR)))
    {
#if EZFB_API_ERRORS
        perror("ezfb ERROR: OPENING FILE DESCRIPTOR FAILED ");
        printf("(%s)\n", fb->dev_name);
#endif
        ret = 0;
    }
    else
    {
        strcpy(fb->dev_name, dev_name);
        fb->dev_num = ezfb_name_to_num(fb->dev_name);
#if EZFB_API_MESSAGES
        printf("FRAME BUFFER %s SUCCESSFULLY OPENED\n", fb->dev_name);
        fflush(stdout);
#endif
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_get_fb_var(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_0
    if(0 > ioctl(fb->fd, FBIOGET_VSCREENINFO, &fb->Var))
    {
#if EZFB_API_ERRORS
        perror("ezfb ERROR: FBIOGET_VSCREENINFO failed ");
        printf("(%s)\n", fb->dev_name);
#endif
    }
    else // ioctl did not fail
    {
        if(!fb->exists)
        {
            fb->bpp_was          = fb->Var.bits_per_pixel ;
            fb->xres_was         = fb->Var.xres           ;
            fb->yres_was         = fb->Var.yres           ;
            fb->xres_virtual_was = fb->Var.xres_virtual   ;
            fb->yres_virtual_was = fb->Var.yres_virtual   ;
            fb->xoffset_was      = fb->Var.xoffset        ;
            fb->yoffset_was      = fb->Var.yoffset        ;
        }
        fb->max_x = fb->Var.xres - 1;
        fb->max_y = fb->Var.yres - 1;
        fb->mid_x = fb->max_x / 2;
        fb->mid_y = fb->max_y / 2;
        ret       = 1;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_fb_var(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(0 > ioctl(fb->fd, FBIOPUT_VSCREENINFO, &fb->Var))
    {
#if EZFB_API_ERRORS
        perror("ezfb ERROR: FBIOPUT_VSCREENINFO failed ");
        printf("(%s)\n", fb->dev_name);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_get_fb_fix(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(0 > ioctl(fb->fd, FBIOGET_FSCREENINFO, &fb->Fix))
    {
#if EZFB_API_ERRORS
        perror("ezfb ERROR: FBIOGET_FSCREENINFO failed ");
        printf("(%s)\n", fb->dev_name);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_get_fb_mem(struct ezfb* fb, int save_old)
{
EZFB_FUNCTION_CALL_1
    if(MAP_FAILED == (fb->data = (u_char*)
                                 mmap( NULL,
                                       fb->Fix.smem_len,
                                       (PROT_READ | PROT_WRITE),
                                       MAP_SHARED,
                                       fb->fd,
                                       0)))
    {
#if EZFB_API_ERRORS
        perror("ezfb ERROR: MEMORY MAP FOR FRAME BUFFER DEVICE FAILED ");
        printf("(%s)\n", fb->dev_name);
#endif
        ret = 0;
    }
    else if(save_old)
    {
        fb->data_was = (u_char*)malloc(fb->Fix.smem_len * sizeof(u_char));
        printf("SAVING CONTENTS OF VIDEO RAM...\na lot of video memory takes a while.\n");
        fflush(stdout);
        memcpy((void*)fb->data_was, (const void*)fb->data, fb->Fix.smem_len); // this can take a while!
#if EZFB_API_MESSAGES
        printf("CONTENTS OF VIDEO RAM SAVED\n");
        fflush(stdout);
#endif
    }
    else
    {
        fb->data_was = NULL; // just to be sure
#if EZFB_API_MESSAGES
        printf("CONTENTS OF VIDEO RAM NOT SAVED\n");
        fflush(stdout);
#endif
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_find_screen(struct ezfb* fb)
{   
    u_int offset;
EZFB_FUNCTION_CALL
    if(fb->exists)
    {
        switch(fb->Var.bits_per_pixel)
        {
            case  1: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset) / 8);
                     break;
            //----------------------------------------------------------------
            case  2: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset) / 4);
                     break;
            //----------------------------------------------------------------
            case  4: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset) / 2);  
                     break;
            //----------------------------------------------------------------
            case  8: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset)    );  
                     break;
            //----------------------------------------------------------------
            case 15: 
            case 16: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset) * 2);  
                     break;
            //----------------------------------------------------------------
            case 24: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset) * 3);  
                     break;
            //----------------------------------------------------------------
            case 32: offset = ((fb->Var.yoffset * fb->Var.xres_virtual + fb->Var.xoffset) * 4);  
                     break;
            //----------------------------------------------------------------
            default: exit(-1);
        }
        fb->screen_uchars  = fb->data + offset;
        fb->screen_uints   = (u_int*)   fb->screen_uchars;
        fb->screen_ushorts = (u_short*) fb->screen_uchars;
    }
EZFB_FUNCTION_RETURN(fb->exists)
}

//############################################################################
// ezfb constructor
// An ezfb object should be initialized like this:
// struct ezfb fb = {0};
//----------------------------------------------------------------------------
int ezfb_init_base(struct ezfb* fb, char* dev_name, int save_old)
{
EZFB_FUNCTION_CALL_1
    srandom(time(NULL));
    ezfb_set_sig_traps();
    if(fb)
    {
        if(valid_ezfb[ezfb_name_to_num(dev_name)] == NULL)
        {
            if(!fb->exists)
            {   
                ret = ezfb_open_fb(fb, dev_name); // sets fb->dev_name, fb->dev_num
                if(ret)
                    ret &= ezfb_get_fb_fix(fb);
                if(ret)
                    ret &= ezfb_get_fb_var(fb);
                if(ret)
                    ret &= ezfb_get_fb_mem(fb, save_old);
                if(ret)
                    ret &= ezfb_init_cmap(fb, save_old);
                if(ret && fb->dev_num == 0)
                    ret &= ezfb_tty_graphics();
                if(ret)
                {
                    fb->exists = 1;
                    ret &= ezfb_find_screen(fb);
                }
                if(ret)
                {
                    valid_ezfb[fb->dev_num] = fb;
                    num_valid_ezfb++;
                    fb->colors.black  = 0x00000000;
                    fb->colors.white  = 0x00ffffff;
                }
#if EZFB_API_MESSAGES
                if(ret)
                {
                    printf("ezfb %s SUCCESSFULLY INITIALIZED\n", fb->dev_name);
                    fflush(stdout);
                }
#endif
            } // if(!fb->exists)
            else
            {
#if EZFB_API_ERRORS
                fprintf(stderr, "ezfb ERROR: ezfb_init_base( %s ) CALLED MORE THAN ONCE!\n", fb->dev_name);
                fflush(stderr);
#endif
                ret = 0;
            }
        } // if(valid_ezfb[ezfb_name_to_num(dev_name)] == NULL)
        else
        {
#if EZFB_API_ERRORS
            fprintf(stderr, "ezfb ERROR: FRAME BUFFER %s ALREADY OPEN BY THIS APPLICATION\n", dev_name);
            fflush(stderr);
#endif
            ret = 0;
        }
    } // if(fb)
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "ezfb ERROR: CAN NOT INITIALIZE NULL POINTER\n");
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init(struct ezfb* fb) // each call attmepts to init next fb
{
EZFB_FUNCTION_CALL_1
    if(num_valid_ezfb <= EZFB_MAX_FB_NUM)
        ret = ezfb_init_num(fb, num_valid_ezfb);
#if EZFB_API_ERRORS
    if(!ret)
    {
        fprintf(stderr, "ezfb ERROR: NO MORE VALID FRAME BUFFER DEVICES\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init_no_save(struct ezfb* fb) // each call attmepts to init next fb
{
EZFB_FUNCTION_CALL_1
    if(num_valid_ezfb <= EZFB_MAX_FB_NUM)
        ret = ezfb_init_num_no_save(fb, num_valid_ezfb);
#if EZFB_API_ERRORS
    if(!ret)
    {
        fprintf(stderr, "ezfb ERROR: NO MORE VALID FRAME BUFFER DEVICES\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init_num(struct ezfb* fb, u_int dev_num)
{
    char dev_name[EZFB_MAX_NAME_LENGTH] = {'\0'};
EZFB_FUNCTION_CALL_1
    if(dev_num >= 0 && dev_num <= EZFB_MAX_FB_NUM)
    {
        sprintf(dev_name, "%s%u", EZFB_DEV_NAME, dev_num);
        if(!(ret = ezfb_init_base(fb, dev_name, 1)))
        {
            sprintf(dev_name, "%s/%d", EZFB_DEV_NAME, dev_num);
            ret = ezfb_init_base(fb, dev_name, 1);
        }
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "ezfb ERROR: %d OUT OF RANGE!\n", dev_num);
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init_num_no_save(struct ezfb* fb, u_int dev_num)
{
    char dev_name[EZFB_MAX_NAME_LENGTH] = {'\0'};
EZFB_FUNCTION_CALL_1
    if(dev_num >= 0 && dev_num <= EZFB_MAX_FB_NUM)
    {
        sprintf(dev_name, "%s%u", EZFB_DEV_NAME, dev_num);
        if(!(ret = ezfb_init_base(fb, dev_name, 0)))
        {
            sprintf(dev_name, "%s/%d", EZFB_DEV_NAME, dev_num);
            ret = ezfb_init_base(fb, dev_name, 0);
        }
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "ezfb ERROR: %d OUT OF RANGE!\n", dev_num);
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init_named(struct ezfb* fb, char* dev_name)
{
EZFB_FUNCTION_CALL_1
    if(!ezfb_init_base(fb, dev_name, 1))
        ret = 0;
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_init_named_no_save(struct ezfb* fb, char* dev_name)
{
EZFB_FUNCTION_CALL_1
    if(!ezfb_init_base(fb, dev_name, 0))
        ret = 0;
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_bpp(struct ezfb* fb, u_int bpp_wanted)
{
EZFB_FUNCTION_CALL_0
    if(fb->exists)
    {
        if(fb->Var.bits_per_pixel == bpp_wanted)
        {
#if EZFB_API_FUNCTION_NIT_PICKING
            fprintf(stderr, "ezfb ERROR: ALREADY AT %d bpp.\n", bpp_wanted);
            fflush(stderr);
#endif
            ret = 1; // error, but return ok
        }
        else
        {
            fb->Var.bits_per_pixel = bpp_wanted;
            ret  = ezfb_set_fb_var (fb);
            ret &= ezfb_get_fb_var (fb);
            if(fb->Var.bits_per_pixel != bpp_wanted) // bpp not supported
            {
#if EZFB_API_ERRORS
                fprintf(stderr, "ezfb ERROR: ezfb_set_bpp(fb, %d) FAILED\n", bpp_wanted);
                fflush(stderr);
#endif
                ret = 0;
            }
            else
            {
                fb->bpp_was_changed   = 1;
                fb->colors.fbcmap.len =   (fb->Var.bits_per_pixel <= 8)
                                        ? (0x01 << fb->Var.bits_per_pixel)
                                        : 0;
                ret &= ezfb_set_cmap        (fb);
                ret &= ezfb_get_cmap        (fb);
                ret &= ezfb_get_color_factors    (fb);
                if(fb->Var.bits_per_pixel <= 8)
                    ret &= ezfb_set_cmap_utility (fb);
                ret &= ezfb_find_black_and_white (fb);
                ret &= ezfb_find_screen          (fb);
            }
        }
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_set_bpp) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_xy_virtual(struct ezfb* fb, u_int xvirt, u_int yvirt)
{
EZFB_FUNCTION_CALL_0
    if(fb->exists)
    {
        fb->Var.xres_virtual = xvirt;
        fb->Var.yres_virtual = yvirt;
        ret  = ezfb_set_fb_var    (fb);
        ret &= ezfb_get_fb_var    (fb);
        if(    (fb->Var.xres_virtual != xvirt)
            || (fb->Var.yres_virtual != yvirt)
          )
        {
#if EZFB_API_ERRORS
            fprintf( stderr,
                     "ezfb ERROR: ezfb_set_xy_virtual(fb, %d, %d) FAILED\n",
                     xvirt, yvirt);
            fflush(stderr);
#endif
            if(    (fb->Var.xres_virtual != fb->xres_virtual_was)
                || (fb->Var.yres_virtual != fb->yres_virtual_was)
              )
            {
                fb->resolution_changed = 1;
#if EZFB_API_MESSAGES
                printf("ezfb %s xres_virtual : %d, yres_virtual : %d\n",
                       fb->dev_name, fb->Var.xres_virtual, fb->Var.yres_virtual);
                fflush(stdout);
#endif
            }
            ret = 0;
        }
        else if(    (fb->Var.xres_virtual != fb->xres_virtual_was)
                 || (fb->Var.yres_virtual != fb->yres_virtual_was)
               )
        {
#if EZFB_API_MESSAGES
            printf("ezfb %s xres_virtual : %d, yres_virtual : %d\n",
                   fb->dev_name, fb->Var.xres_virtual, fb->Var.yres_virtual);
            fflush(stdout);
#endif
            fb->resolution_changed = 1;
        }
#if EZFB_API_MESSAGES
        else
        {
            printf("ezfb %s xres_virtual : %d, yres_virtual : %d : no change\n",
                   fb->dev_name, fb->Var.xres_virtual, fb->Var.yres_virtual);
            fflush(stdout);
        }
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_set_xy_virtual) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_set_xy_offsets(struct ezfb* fb, int xoff, int yoff)
{
EZFB_FUNCTION_CALL_0
    if(fb->exists)
    {
        if(    (xoff >= 0)
            && (xoff <= (int)(fb->Var.xres_virtual - fb->Var.xres))
          )
            fb->Var.xoffset = xoff;
        //--------------------------------------------------------------------
        if(    (yoff >= 0)
            && (yoff <= (int)(fb->Var.yres_virtual - fb->Var.yres))
          )
            fb->Var.yoffset = yoff;
        //--------------------------------------------------------------------
        ret  = ezfb_set_fb_var    (fb);
        ret &= ezfb_get_fb_var    (fb);
        ret &= ezfb_set_cmap      (fb);
        //--------------------------------------------------------------------
        if(    ((int)fb->Var.xoffset != xoff)
            || ((int)fb->Var.yoffset != yoff) // we didn't get what we wanted
          )
        {
#if EZFB_API_ERRORS
            fprintf(stderr, "ezfb ERROR: ezfb_set_xy_offsets(fb, %d, %d) FAILED\n", xoff, yoff);
            fflush(stderr);
#endif            
        }
        //--------------------------------------------------------------------
        if(    (fb->Var.xoffset != fb->xoffset_was)
            || (fb->Var.yoffset != fb->yoffset_was) // we changed something
          )
        {
#if EZFB_API_FUNCTION_NIT_PICKING
            printf("ezfb %s xoffset : %d, yoffset :  %d\n", fb->dev_name, fb->Var.xoffset, fb->Var.yoffset);
            fflush(stdout);
#endif
            fb->resolution_changed = 1;
        }
        //--------------------------------------------------------------------
#if EZFB_API_FUNCTION_NIT_PICKING
        else // we didn't change anything
        {
            printf("ezfb %s xoffset : %d, yoffset :  %d : no change\n", fb->dev_name, fb->Var.xoffset, fb->Var.yoffset);
            fflush(stdout);
            ret = 0;
        }
#endif
    }
    //------------------------------------------------------------------------
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_set_xy_offsets) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
//int ezfb_set_resolution(struct ezfb* fb, int new_xres, int new_yres)
//{       //TODO: make this work!
//EZFB_FUNCTION_CALL_1
//    if(fb->exists)
//    {
//        fb->Var.xres = new_xres;
//        fb->Var.yres = new_yres;
//        ret &= ezfb_set_fb_var(fb);
//        ret &= ezfb_get_fb_var(fb);
//        if(ret) fb->resolution_changed = 1;
//    }
//    else
//    {
//#if EZFB_API_ERRORS
//        fprintf(stderr, "ezfb ERROR: (ezfb_set_resolution) FB NOT INITIALIZED");
//        fflush(stderr);
//#endif
//        ret = 0;
//    }
//EZFB_FUNCTION_RETURN(ret)
//}

//############################################################################
int ezfb_copy_old_fb(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(fb->exists && fb->data_was)
    {   // TODO: deal with x & y offsets that might have changed
        memcpy((void*)fb->data, (const void*)fb->data_was, fb->Fix.smem_len);
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "ezfb ERROR: fb->data_was NEVER SAVED!\n");
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_clear(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_1
    if(fb->exists)
    {
        memset((void*)fb->data, 0x00, fb->Fix.smem_len);
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "ezfb ERROR: (ezfb_clear) FB NOT INITIALIZED\n");
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_clear_screen(struct ezfb* fb)
{
    u_int bytes, y;
EZFB_FUNCTION_CALL_1
    if(fb->exists)
    {
        ezfb_find_screen(fb);
        switch(fb->Var.bits_per_pixel)
        {
            case  1: bytes = fb->Var.xres / 8;
                     for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + ((y * fb->Var.xres_virtual) / 8))
                                 , 0x00, bytes);
                     break;
            //---------------------------------------------------------------- 
            case  2: for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + ((y * fb->Var.xres_virtual) / 4))
                                 , 0x00, fb->Var.xres / 4);
                     break;
            //---------------------------------------------------------------- 
            case  4: for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + ((y * fb->Var.xres_virtual) / 2))
                                 , 0x00, fb->Var.xres / 2);
                     break;
            //---------------------------------------------------------------- 
            case  8: for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + (y * fb->Var.xres_virtual))
                                 , 0x00, fb->Var.xres);
                     break;
            //---------------------------------------------------------------- 
            case 15: 
            case 16: for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + (2 * y * fb->Var.xres_virtual))
                                 , 0x00, fb->Var.xres * 2);
                     break;
            //---------------------------------------------------------------- 
            case 24: for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + (3 * y * fb->Var.xres_virtual))
                                 , 0x00, fb->Var.xres * 3);
                     break;
            //---------------------------------------------------------------- 
            case 32: for(y = 1; y < fb->Var.yres; y++)
                         memset(   (void*)(fb->screen_uchars
                                 + (4 * y * fb->Var.xres_virtual))
                                 , 0x00, fb->Var.xres * 4);
                     break;
            //---------------------------------------------------------------- 
            default: exit(-1);
        }
    }
    else
    {
#if EZFB_API_ERRORS
        fprintf(stderr, "ezfb ERROR: (ezfb_clear_screen) FB NOT INITIALIZED\n");
        fflush(stderr);
#endif
        ret = 0;
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
// ezfb destructor
//----------------------------------------------------------------------------
int ezfb_release(struct ezfb* fb)
{
EZFB_FUNCTION_CALL_0
    if(fb)
    {
        if(fb->exists)
        {
            ret = 1;
#if EZFB_API_MESSAGES
            printf("ezfb %s shutting down\n", fb->dev_name);
            fflush(stdout);
#endif
            if(fb->bpp_was_changed)
            {
                ret &= ezfb_set_bpp(fb, fb->bpp_was);
#if EZFB_API_MESSAGES
                if(ret)
                {
                    printf("ezfb %s bpp set back to %d\n", fb->dev_name, fb->Var.bits_per_pixel);
                    fflush(stdout);
                }
#endif
            }
            if(fb->resolution_changed)
            {
                ret &= ezfb_set_xy_virtual(fb, fb->xres_virtual_was, fb->yres_virtual_was);
                ret &= ezfb_set_xy_offsets(fb, fb->xoffset_was     , fb->yoffset_was     );
                //ret &= ezfb_set_resolution(fb, fb->xres_was,         fb->yres_was        );
#if EZFB_API_MESSAGES
                if(ret)
                {
                    printf("ezfb %s xres_virtual set back to %d\n", fb->dev_name, fb->Var.xres_virtual);
                    fflush(stdout);
                    printf("ezfb %s yres_virtual set back to %d\n", fb->dev_name, fb->Var.yres_virtual);
                    fflush(stdout);
                    printf("ezfb %s xoffset set back to %d\n"     , fb->dev_name, fb->Var.xoffset     );
                    fflush(stdout);
                    printf("ezfb %s yoffset set back to %d\n",      fb->dev_name, fb->Var.yoffset     );
                    fflush(stdout);
                }
#endif
            }
            if(fb->data_was)
            {
                memcpy((void*)fb->data, (const void*)fb->data_was, fb->Fix.smem_len); // TODO reset the x & y offsets
                free(fb->data_was);
                fb->data_was = NULL;
#if EZFB_API_MESSAGES
                printf("ezfb %s memory contents restored\n", fb->dev_name);
                fflush(stdout);
#endif
            }
            ret &= ezfb_release_cmap(fb);

            if(fb->dev_num == 0)
                ret &= ezfb_tty_text();

            close(fb->fd);
#if EZFB_API_MESSAGES
            printf("ezfb %s file descriptor closed\n", fb->dev_name);
            fflush(stdout);
#endif
            fb->exists = 0;
            valid_ezfb[fb->dev_num] = NULL;
            num_valid_ezfb--;
#if EZFB_API_MESSAGES
            if(ret)
            {
                printf("ezfb %s SUCCESSFULLY RELEASED\n", fb->dev_name);
                fflush(stdout);
            }
#endif
        }
#if EZFB_API_ERRORS
        else
        {
            fprintf(stderr, "ezfb ERROR: (ezfb_release) FB NOT INITIALIZED\n");
            fflush(stderr);
        }
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: RELEASE CALLED ON NULL POINTER\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_dump_fb_fix_screeninfo(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->exists)
    {
#if EZFB_API_MESSAGES
        printf("\n"                                                                   );
        printf("      FB FIXED SCREEN INFO:\n"                                        );
        printf("      fb->Fix.id               = %s\n"  ,        fb->Fix.id           ); // identification string eg "TT Builtin"
        printf("      fb->Fix.smem_start       = 0x%X\n", (u_int)fb->Fix.smem_start   ); // Start of frame buffer mem (physical address)
        printf("      fb->Fix.smem_len         = %u\n"  ,        fb->Fix.smem_len     ); // Length of frame buffer mem
        printf("      fb->Fix.type             = %u\n"  ,        fb->Fix.type         ); // see FB_TYPE_*        */
        printf("      fb->Fix.type_aux         = %u\n"  ,        fb->Fix.type_aux     ); // Interleave for interleaved Planes
        printf("      fb->Fix.visual           = %u\n"  ,        fb->Fix.visual       ); // see FB_VISUAL_*        */
        printf("      fb->Fix.xpanstep         = %u\n"  ,        fb->Fix.xpanstep     ); // zero if no hardware panning
        printf("      fb->Fix.ypanstep         = %u\n"  ,        fb->Fix.ypanstep     ); // zero if no hardware panning
        printf("      fb->Fix.ywrapstep        = %u\n"  ,        fb->Fix.ywrapstep    ); // zero if no hardware ywrap
        printf("      fb->Fix.line_length      = %u\n"  ,        fb->Fix.line_length  ); // length of a line in bytes
        printf("      fb->Fix.mmio_start       = 0x%X\n", (u_int)fb->Fix.mmio_start   ); // Start of Memory Mapped I/O (physical address)
        printf("      fb->Fix.mmio_len         = %u\n"  ,        fb->Fix.mmio_len     ); // Length of Memory Mapped I/O
        printf("      fb->Fix.accel            = %u\n"  ,        fb->Fix.accel        ); // Type of acceleration available
        printf("      fb->Fix.reserved[0]      = %u\n"  ,        fb->Fix.reserved[0]  ); // Reserved for future compatibility
        printf("      fb->Fix.reserved[1]      = %u\n"  ,        fb->Fix.reserved[1]  ); // Reserved for future compatibility
        printf("      fb->Fix.reserved[2]      = %u\n"  ,        fb->Fix.reserved[2]  ); // Reserved for future compatibility
        printf("\n");
        fflush(stdout);
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_dump_fb_fix_screeninfo) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->exists)
}

//############################################################################
int ezfb_dump_fb_var_screeninfo(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->exists)
    {
        ezfb_get_fb_var(fb);
#if EZFB_API_MESSAGES
        printf("\n"                                                                );
        printf("      FB VARIABLE SCREEN INFO:\n"                                  );
        printf("      fb->Var.xres             = %u\n"  , fb->Var.xres             ); // visible resolution
        printf("      fb->Var.yres             = %u\n"  , fb->Var.yres             );
        printf("      fb->Var.xres_virtual     = %u\n"  , fb->Var.xres_virtual     ); // virtual resolution
        printf("      fb->Var.yres_virtual     = %u\n"  , fb->Var.yres_virtual     );
        printf("      fb->Var.xoffset          = %u\n"  , fb->Var.xoffset          ); // offset from virtual to visible
        printf("      fb->Var.yoffset          = %u\n"  , fb->Var.yoffset          ); // resolution
        printf("      fb->Var.bits_per_pixel   = %u\n"  , fb->Var.bits_per_pixel   ); // guess what
        printf("      fb->Var.grayscale        = %u\n"  , fb->Var.grayscale        ); // != 0 Graylevels instead of colors
        printf("      fb->Var.red.offset       = %u\n"  , fb->Var.red.offset       ); // bitfield in fb mem if true color,
        printf("      fb->Var.red.length       = %u\n"  , fb->Var.red.length       ); // else only length is significant
        printf("      fb->Var.red.msb_right    = %u\n"  , fb->Var.red.msb_right    );
        printf("      fb->Var.green.offset     = %u\n"  , fb->Var.green.offset     );
        printf("      fb->Var.green.length     = %u\n"  , fb->Var.green.length     );
        printf("      fb->Var.green.msb_right  = %u\n"  , fb->Var.green.msb_right  );
        printf("      fb->Var.blue.offset      = %u\n"  , fb->Var.blue.offset      );
        printf("      fb->Var.blue.length      = %u\n"  , fb->Var.blue.length      );
        printf("      fb->Var.blue.msb_right   = %u\n"  , fb->Var.blue.msb_right   );
        printf("      fb->Var.transp.offset    = %u\n"  , fb->Var.transp.offset    ); // transparency
        printf("      fb->Var.transp.length    = %u\n"  , fb->Var.transp.length    );
        printf("      fb->Var.transp.msb_right = %u\n"  , fb->Var.transp.msb_right );
        printf("      fb->Var.nonstd           = %u\n"  , fb->Var.nonstd           ); // != 0 Non standard pixel format
        printf("      fb->Var.activate         = %u\n"  , fb->Var.activate         ); // see FB_ACTIVATE_*
        printf("      fb->Var.height           = %u\n"  , fb->Var.height           ); // height of picture in mm
        printf("      fb->Var.width            = %u\n"  , fb->Var.width            ); // width of picture in mm
        printf("      fb->Var.accel_flags      = %u\n"  , fb->Var.accel_flags      ); // acceleration flags (hints)
        printf("      fb->Var.pixclock         = %u\n"  , fb->Var.pixclock         ); // pixel clock in ps (pico seconds)
        printf("      fb->Var.left_margin      = %u\n"  , fb->Var.left_margin      ); // time from sync to picture
        printf("      fb->Var.right_margin     = %u\n"  , fb->Var.right_margin     ); // time from picture to sync
        printf("      fb->Var.upper_margin     = %u\n"  , fb->Var.upper_margin     ); // time from sync to picture
        printf("      fb->Var.lower_margin     = %u\n"  , fb->Var.lower_margin     );
        printf("      fb->Var.hsync_len        = %u\n"  , fb->Var.hsync_len        ); // length of horizontal sync
        printf("      fb->Var.vsync_len        = %u\n"  , fb->Var.vsync_len        ); // length of vertical sync
        printf("      fb->Var.sync             = %u\n"  , fb->Var.sync             ); // see FB_SYNC_*
        printf("      fb->Var.vmode            = %u\n"  , fb->Var.vmode            ); // see FB_VMODE_*
        printf("      fb->Var.reserved[0]      = %u\n"  , fb->Var.reserved[0]      ); // Reserved for future compatibility
        printf("      fb->Var.reserved[1]      = %u\n"  , fb->Var.reserved[1]      ); // Reserved for future compatibility
        printf("      fb->Var.reserved[2]      = %u\n"  , fb->Var.reserved[2]      ); // Reserved for future compatibility
        printf("      fb->Var.reserved[3]      = %u\n"  , fb->Var.reserved[3]      ); // Reserved for future compatibility
        printf("      fb->Var.reserved[4]      = %u\n"  , fb->Var.reserved[4]      ); // Reserved for future compatibility
        printf("      fb->Var.reserved[5]      = %u\n"  , fb->Var.reserved[5]      ); // Reserved for future compatibility
        printf("\n");
        fflush(stdout);
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_dump_fb_var_screeninfo) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->exists)
}

//############################################################################
int ezfb_dump(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->exists)
    {
#if EZFB_API_MESSAGES
        printf("      EZFB GLOBAL PARAMETERS:\n"                                  );
        printf("      fb->fd                 = %d\n"  , fb->fd                    );
        printf("      fb->dev_num            = %u\n"  , fb->dev_num               );
        printf("      fb->dev_name           = %s\n"  , fb->dev_name              );
        printf("      fb->exists             = %u\n"  , fb->exists                );
        printf("      fb->resolution_changed = %u\n"  , fb->resolution_changed    );
        printf("      fb->bpp_was_changed    = %u\n"  , fb->bpp_was_changed       );
        printf("      fb->xres_was           = %u\n"  , fb->xres_was              );
        printf("      fb->yres_was           = %u\n"  , fb->yres_was              );
        printf("      fb->xres_virtual_was   = %u\n"  , fb->xres_virtual_was      );
        printf("      fb->yres_virtual_was   = %u\n"  , fb->yres_virtual_was      );
        printf("      fb->xoffset_was        = %u\n"  , fb->xoffset_was           );
        printf("      fb->yoffset_was        = %u\n"  , fb->yoffset_was           );
        printf("      fb->bpp_was            = %u\n"  , fb->bpp_was               );
        printf("      fb->data               = 0x%X\n", (u_int)fb->data           );
        printf("      fb->screen_uchars      = 0x%X\n", (u_int)fb->screen_uchars  );
        printf("      fb->screen_ushorts     = 0x%X\n", (u_int)fb->screen_ushorts );
        printf("      fb->screen_uints       = 0x%X\n", (u_int)fb->screen_uints   );
        printf("      fb->data_was           = 0x%X\n", (u_int)fb->data_was       );
        printf("      fb->max_x              = %u\n"  , fb->max_x                 );
        printf("      fb->max_y              = %u\n"  , fb->max_y                 );
        printf("      fb->mid_x              = %u\n"  , fb->mid_x                 );
        printf("      fb->mid_y              = %u\n"  , fb->mid_y                 );
        printf("\n");
        fflush(stdout);
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_dump) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(1)
}

//############################################################################
int ezfb_dump_screen_map(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->exists)
    {
#if EZFB_API_MESSAGES
        u_int i;
        printf("\n");
        printf("FB SCREEN MAP %d x %d : %d bytes:", fb->Var.xres, fb->Var.yres, fb->Var.xres * fb->Var.yres);
        for(i = 0; i < fb->Var.xres * fb->Var.yres; i++)
        {
            if(0 == (i % fb->Var.xres))
                printf("\n");
            printf(" %2X" , fb->screen_uchars[i]);
            fflush(stdout);
        }
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_dump_screen_map) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->exists)
}

//############################################################################
int ezfb_dump_virtual_map(struct ezfb* fb)
{
EZFB_FUNCTION_CALL
    if(fb->exists)
    {
#if EZFB_API_MESSAGES
        u_int i;
        printf("\n");
        printf("FB VIRTUAL PIXEL MAP %d x %d : %d bytes:", fb->Var.xres_virtual, fb->Var.yres_virtual, fb->Var.xres_virtual * fb->Var.yres_virtual);
        for(i = 0; i < fb->Fix.smem_len; i++)
        {
            if(0 == (i % fb->Var.xres_virtual))
                printf("\n");
            printf(" %2X" , fb->data[i]);
            fflush(stdout);
        }
#endif
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_dump_virtual_map) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(fb->exists)
}

//############################################################################
int ezfb_binfile_screen_map(struct ezfb* fb, char* file_name)
{
EZFB_FUNCTION_CALL_0
    if(fb->exists)
    {
        int binfile_fd;
        if(0 > (binfile_fd = open(file_name, (O_WRONLY | O_CREAT), (S_IRWXU | S_IRGRP | S_IROTH)))) // mode 744
        {
#if EZFB_API_ERRORS
            perror("ezfb ERROR: opening output file failed\n");
#endif
            ret = 0;
        }
        else
        {
            ret = 1;
            write(binfile_fd, (void*)fb->screen_uchars, fb->Var.xres * fb->Var.yres);
            close(binfile_fd);
        }
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_binfile_screen_map) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_binfile_virtual_map(struct ezfb* fb, char* file_name)
{
EZFB_FUNCTION_CALL_0
    if(fb->exists)
    {
        int binfile_fd;
        if(0 > (binfile_fd = open(file_name, (O_WRONLY | O_CREAT), (S_IRWXU | S_IRGRP | S_IROTH)))) // mode 744
        {
#if EZFB_API_ERRORS
            perror("ezfb ERROR: opening output file failed\n");
#endif
            ret = 0;
        }
        else
        {
            ret = 1;
            write(binfile_fd, (void*)fb->data, fb->Fix.smem_len);
            close(binfile_fd);
        }
    }
#if EZFB_API_ERRORS
    else
    {
        fprintf(stderr, "ezfb ERROR: (ezfb_binfile_virtual_map) FB NOT INITIALIZED\n");
        fflush(stderr);
    }
#endif
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
float ezfb_random_01() // returns a float between 0 & 1.
{
    return ((float)(rand()) / (float)RAND_MAX);
}

//############################################################################
float ezfb_random_neg_to_pos_1() // returns a float between -1 & 1.
{
    return sin(rand());
}

//############################################################################
void ezfb_set_sig_traps()
{
EZFB_FUNCTION_CALL
    static int been_done = 0;
    if(!been_done)
    {
        signal(SIGTERM, ezfb_terminate);
        signal(SIGINT,  ezfb_terminate);
        signal(SIGQUIT, ezfb_terminate);
        signal(SIGHUP,  ezfb_terminate);
        signal(SIGTSTP, ezfb_terminate);
        signal(SIGTTIN, ezfb_terminate);
        signal(SIGFPE,  ezfb_terminate);
        signal(SIGILL,  ezfb_terminate);
        signal(SIGSEGV, ezfb_terminate);
        signal(SIGBUS,  ezfb_terminate);
        signal(SIGABRT, ezfb_terminate);
        signal(SIGSYS,  ezfb_terminate);
        been_done = 1;
#if EZFB_API_MESSAGES
        printf("TERMINATION SIGNAL TRAPS SET\n");
        fflush(stdout);
#endif
    }
EZFB_FUNCTION_RETURN_
}

//############################################################################
void ezfb_terminate(int signum)
{

    int i;
    for(i = 0; i <= EZFB_MAX_FB_NUM; i++)
    {
        if(valid_ezfb[i])
        {
#if EZFB_API_MESSAGES
            printf("FRAME BUFFER %s TERMINATING... ", valid_ezfb[i]->dev_name);
            fflush(stdout);
#endif
            ezfb_release(valid_ezfb[i]);
#if EZFB_API_MESSAGES
            printf("DONE\n");
            fflush(stdout);
#endif
        }
    }
    printf("\n");
    fflush(stdout);
    exit(0);
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################

