//############################################################################
//
// EZFB ~ Linux Frame Buffer API ~
//
// by James Lehman
// james@akrobiz.com
//
// began: Feb. 2001
//
// EZFB is intended to give C programmers an easy-to-use library of functions
// for drawing points, lines and strings of characters into the Linux frame
// buffer. It also provides a means of displaying bitmap images on any portion
// of the screen and capturing the contents of any portion of the screen to
// a standard bitmap graphics file. All of this is embodied in a set of
// functions that are polymorphic with regard to the characteristics of the
// frame buffer and the bitmap files used. In other words, it makes no
// difference what the screen height or width is or what the color depth is,
// all of the function calls are the same and will have appropriate and
// predictable effects.
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ezfb_colors.h"
#include "ezfb_pixels.h"

//############################################################################
int ezfb_put_pixel(struct ezfb* fb, u_int  x, u_int  y, u_int  c)
{
EZFB_NIT_PICKING_FUNCTION_CALL_0
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        switch(fb->Var.bits_per_pixel)
        {
            case  1: break;
            //----------------------------------------------------------------
            case  2: break;
            //----------------------------------------------------------------
            case  4: {
                         int i = (y * fb->Var.xres_virtual + x) / 2;
                         fb->screen_uchars[i] = (x & 0x01) // odd
                                                 ? ((fb->screen_uchars[i] & 0x0F) | ((c & 0x0F) << 4))
                                                 : ((fb->screen_uchars[i] & 0xF0) |  (c & 0x0F)      );
                     }
                     break;
            //----------------------------------------------------------------
            case  8: fb->screen_uchars[y * fb->Var.xres_virtual + x] = c;
                     break;
            //----------------------------------------------------------------
            case 15:
            case 16: fb->screen_ushorts[y * fb->Var.xres_virtual + x] = c;
                     break;
            //----------------------------------------------------------------
            case 24: {
                         int i = (y * fb->Var.xres_virtual + x) * 3;
                         fb->screen_uchars[i++] = (u_char)ezfb_get_b_from_rgb(fb, c);
                         fb->screen_uchars[i++] = (u_char)ezfb_get_g_from_rgb(fb, c);
                         fb->screen_uchars[i  ] = (u_char)ezfb_get_r_from_rgb(fb, c);
                     }
                     break;
            //----------------------------------------------------------------
            case 32: fb->screen_uints[y * fb->Var.xres_virtual + x] = c;
                     break;
            //----------------------------------------------------------------
            default: ;
        }
        ret = 1;
    }
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_put_r_in_pixel(struct ezfb* fb, u_int  x, u_int  y, u_short r)
{
EZFB_NIT_PICKING_FUNCTION_CALL_0
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        switch(fb->Var.bits_per_pixel)
        {
            default: ;
            //----------------------------------------------------------------
            case 15:
            case 16: {
                         int i = y * fb->Var.xres_virtual + x;
                         fb->screen_ushorts[i]
                           = (fb->screen_ushorts[i] & (~fb->colors.bit_mask_red << fb->Var.red.offset))
                           | (r & fb->colors.bit_mask_red) << fb->Var.red.offset;
                     }
                     ret = 1;
            //----------------------------------------------------------------
            case 24: fb->screen_uchars[(y * fb->Var.xres_virtual + x) * 3 + 2] = (u_char)r;
                     ret = 1;
            //----------------------------------------------------------------
            case 32: fb->screen_uchars[(y * fb->Var.xres_virtual + x) * 4 + 2] = (u_char)r;
                     ret = 1;
        }
    }
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_put_g_in_pixel(struct ezfb* fb, u_int  x, u_int  y, u_short g)
{
EZFB_NIT_PICKING_FUNCTION_CALL_0
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        switch(fb->Var.bits_per_pixel)
        {
            default: ;
            //----------------------------------------------------------------
            case 15:
            case 16: {
                         int i = y * fb->Var.xres_virtual + x;
                         fb->screen_ushorts[i]
                           = (fb->screen_ushorts[i] & (~fb->colors.bit_mask_green << fb->Var.green.offset))
                           | (g & fb->colors.bit_mask_green) << fb->Var.green.offset;
                     }
                     return 1;
            //----------------------------------------------------------------
            case 24: fb->screen_uchars[(y * fb->Var.xres_virtual + x) * 3 + 1] = (u_char)g;
                     return 1;
            //----------------------------------------------------------------
            case 32: fb->screen_uchars[(y * fb->Var.xres_virtual + x) * 4 + 1] = (u_char)g;
                     return 1;
        }
    }
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_put_b_in_pixel(struct ezfb* fb, u_int  x, u_int  y, u_short b)
{
EZFB_NIT_PICKING_FUNCTION_CALL_0
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        switch(fb->Var.bits_per_pixel)
        {
            default: ;
            //----------------------------------------------------------------
            case 15:
            case 16: {
                         int i = y * fb->Var.xres_virtual + x;
                         fb->screen_ushorts[i]
                           = (fb->screen_ushorts[i] & (~fb->colors.bit_mask_blue << fb->Var.blue.offset))
                           | (b & fb->colors.bit_mask_blue) << fb->Var.blue.offset;
                     }
                     ret = 1;
            //----------------------------------------------------------------
            case 24: fb->screen_uchars[(y * fb->Var.xres_virtual + x) * 3] = (u_char)b;
                     ret = 1;
            //----------------------------------------------------------------
            case 32: fb->screen_uchars[(y * fb->Var.xres_virtual + x) * 4] = (u_char)b;
                     ret = 1;
        }
    }
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}

//############################################################################
u_int ezfb_get_pixel(struct ezfb* fb, u_int  x, u_int  y)
{
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        switch(fb->Var.bits_per_pixel)
        {
            case  1: return 0;
            //----------------------------------------------------------------
            case  2: return 0;
            //----------------------------------------------------------------
            case  4: {
                        int i = (y * fb->Var.xres_virtual + x) / 2;
                        return (u_int)( (x & 0x01) // odd
                                          ? ((fb->screen_uchars[i] & 0xF0) >> 4)
                                          :  (fb->screen_uchars[i] & 0x0F)       );
                     }
            //----------------------------------------------------------------
            case  8: return (u_int)fb->screen_uchars[y * fb->Var.xres_virtual + x];
            //----------------------------------------------------------------
            case 15:
            case 16: return (u_int)fb->screen_ushorts[y * fb->Var.xres_virtual + x];
            //----------------------------------------------------------------
            case 24: {
                         int i = (y * fb->Var.xres_virtual + x) * 3;
                         return   (fb->screen_uchars[i    ] << 16)
                                | (fb->screen_uchars[i + 1] <<  8)
                                |  fb->screen_uchars[i + 2]        ;
                     }
            //----------------------------------------------------------------
            case 32: return fb->screen_uints[y * fb->Var.xres_virtual + x];
            //----------------------------------------------------------------
        }
    }
    return 0;
}

//############################################################################
u_short ezfb_get_r_from_pixel(struct ezfb* fb, u_int  x, u_int  y)
{
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        if(fb->Var.bits_per_pixel <= 8)
            return fb->colors.red[ezfb_get_pixel(fb, x, y)];
        return ezfb_get_r_from_rgb(fb, ezfb_get_pixel(fb, x, y));
    }
    return 0;
}

//############################################################################
u_short ezfb_get_g_from_pixel(struct ezfb* fb, u_int  x, u_int  y)
{
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        if(fb->Var.bits_per_pixel <= 8)
            return fb->colors.green[ezfb_get_pixel(fb, x, y)];
        return ezfb_get_g_from_rgb(fb, ezfb_get_pixel(fb, x, y));
    }
    return 0;
}

//############################################################################
u_short ezfb_get_b_from_pixel(struct ezfb* fb, u_int  x, u_int  y)
{
    if(fb->exists && (x < fb->Var.xres_virtual) && (y < fb->Var.yres))
    {
        if(fb->Var.bits_per_pixel <= 8)
            return fb->colors.blue[ezfb_get_pixel(fb, x, y)];
        return ezfb_get_b_from_rgb(fb, ezfb_get_pixel(fb, x, y));
    }
    return 0;
}

//############################################################################
int ezfb_put_line(struct ezfb* fb, int x1, int y1, int x2, int y2, int c)
{
    int dx  = x2 - x1;
    int dy  = y2 - y1;
    int n, b;
    float m;
EZFB_NIT_PICKING_FUNCTION_CALL_1
    if(0 == dx)
    {
        if(dy < 0)
            for(n = y1; n >= y2; n--)
                ret &= ezfb_put_pixel(fb, x1, n, c);
        else
            for(n = y1; n <= y2; n++)
                ret &= ezfb_put_pixel(fb, x1, n, c);
    }
    else if(0 == dy)
    {
        if(dx < 0)
            for(n = x1; n >= x2; n--)
                ret &= ezfb_put_pixel(fb, n, y1, c);
        else
            for(n = x1; n <= x2; n++)
                ret &= ezfb_put_pixel(fb, n, y1, c);
    }
    else if(abs(dx) >= abs(dy))
    {
        m = ((float)dy) / ((float)dx);
        b = (int)(y1 - m * x1);
        if(dx < 0)
            for(n = x1; n >= x2; n--)
                ret &= ezfb_put_pixel(fb, n, (u_int)floor(n * m + b + 0.5), c);
        else
            for(n = x1; n <= x2; n++)
                ret &= ezfb_put_pixel(fb, n, (u_int)floor(n * m + b + 0.5), c);
    }
    else
    {
        m = ((float)dx) / ((float)dy);
        b = (int)(x1 - m * y1);
        if(dy < 0)
            for(n = y1; n >= y2; n--)
                ret &= ezfb_put_pixel(fb, (u_int)floor(n * m + b + 0.5), n, c);
        else
            for(n = y1; n <= y2; n++)
                ret &= ezfb_put_pixel(fb, (u_int)floor(n * m + b + 0.5), n, c);
    }
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_put_ray(struct ezfb* fb, int x, int y, float r, float t, int c)
{
    return ezfb_put_line(   fb
                    , x
                    , y
                    , x + (u_int)floor((cos(t) * r) + 0.5)
                    , y + (u_int)floor((sin(t) * r) + 0.5)
                    , c);
}


//############################################################################
int ezfb_put_rectangle(struct ezfb* fb, int x1, int y1, int x2, int y2, int c, int filled)
{
EZFB_NIT_PICKING_FUNCTION_CALL_1
    if(filled)
    {
        int y, step;
        step = (y1 <= y2) ? 1 : -1;
        if((x1 != x2) && (y1 != y2))
            for(y = y1; y <= y2; y += step)
                ret &= ezfb_put_line(fb, x1, y, x2, y, c);
        else if(x1 != x2)
            ret &= ezfb_put_line(fb, x1, y1, x2, y1, c);
        else
            ret &= ezfb_put_pixel(fb, x1, y1, c);
    }
    else
    {
        if((x1 != x2) && (y1 != y2))
        {
            ret &= ezfb_put_line(fb, x1, y1, x2, y1, c);
            ret &= ezfb_put_line(fb, x1, y2, x2, y2, c);
            ret &= ezfb_put_line(fb, x1, y1, x1, y2, c);
            ret &= ezfb_put_line(fb, x2, y1, x2, y2, c);
        }
        else if(x1 != x2)
            ret &= ezfb_put_line(fb, x1, y1, x2, y1, c);
        else
            ret &= ezfb_put_pixel(fb, x1, y1, c);
    }
EZFB_NIT_PICKING_FUNCTION_RETURN(ret)
}
//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################

