//############################################################################
//
// EZFB ~ Linux Frame Buffer API ~
//
// by James Lehman
// james@akrobiz.com
//
// began: Feb. 2001
//
// EZFB is intended to give C programmers an easy-to-use library of functions
// for drawing points, lines and strings of characters into the Linux frame
// buffer. It also provides a means of displaying bitmap images on any portion
// of the screen and capturing the contents of any portion of the screen to
// a standard bitmap graphics file. All of this is embodied in a set of
// functions that are polymorphic with regard to the characteristics of the
// frame buffer and the bitmap files used. In other words, it makes no
// difference what the screen height or width is or what the color depth is,
// all of the function calls are the same and will have appropriate and
// predictable effects.
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ezfb_tty.h"

//############################################################################
struct termios saved_termios         ;
int            stdin_was             ;
int            stdin_was_changed = 0 ;
int            tty                   ;
int            tty_mode_was          ;

//############################################################################
int ezfb_tty_graphics()
{
EZFB_FUNCTION_CALL_1
    if(0 > (tty = open(EZFB_TTY_DEVICE, O_RDWR)))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: " EZFB_TTY_DEVICE " open failed");
#endif
        ret = 0;
    }
    else if(0 > ioctl(tty, KDGETMODE, &tty_mode_was))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: " EZFB_TTY_DEVICE " get mode failed");
#endif
        ret = 0;
    }
    else if(    (KD_GRAPHICS != tty_mode_was)
             && (0 > ioctl(tty, KDSETMODE, KD_GRAPHICS)))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: " EZFB_TTY_DEVICE " set graphics failed");
#endif
        close(tty);
        ret = 0;
    }
    else
    {
#if EZFB_API_MESSAGES
        printf("tty mode was ");
        switch(tty_mode_was)
        {
            case KD_TEXT     :  printf("KD_TEXT"     ); break;
            case KD_GRAPHICS :  printf("KD_GRAPHICS" ); break;
            default          :  printf("IMPOSSIBLE!" ); break;
        }
        printf("\n");
        printf("tty mode set to KD_GRAPHICS\n");
        fflush(stdout);
#endif
        close(tty);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_tty_text()
{
EZFB_FUNCTION_CALL_1
    if(0 > (tty = open(EZFB_TTY_DEVICE, O_RDWR)))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: " EZFB_TTY_DEVICE " open failed");
#endif
        ret = 0;
    }
    else if(0 > ioctl(tty, KDSETMODE, tty_mode_was))
    {
#if EZFB_API_ERRORS
        perror("\nezfb ERROR: " EZFB_TTY_DEVICE " reset mode failed");
#endif
        close(tty);
        ret = 0;
    }
    else
    {
#if EZFB_API_MESSAGES
        printf("tty mode set to ");
        switch(tty_mode_was)
        {
            case KD_TEXT     :  printf("KD_TEXT"     ); break;
            case KD_GRAPHICS :  printf("KD_GRAPHICS" ); break;
            default          :  printf("IMPOSSIBLE!" ); break;
        }
        printf("\n");
        fflush(stdout);
#endif
        close(tty);
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
void ezfb_restore_input()
{
EZFB_FUNCTION_CALL
    if(stdin_was_changed)
    {
        tcsetattr(STDIN_FILENO, TCSANOW, &saved_termios);
        fcntl(fileno(stdin), F_SETFL, stdin_was);        
        stdin_was_changed = 0;
    }
EZFB_FUNCTION_RETURN_
}

//############################################################################
int ezfb_set_input_hot()
{
EZFB_FUNCTION_CALL_1
    if((ret = !stdin_was_changed))
    {
        struct termios new_termios;
        if(!isatty(STDIN_FILENO))
        {
            perror("\nezfb ERROR: not a terminal");
            ret = 0;
        }
        else
        {
            tcgetattr(STDIN_FILENO, &saved_termios);
            atexit(ezfb_restore_input);
            stdin_was = fcntl(fileno(stdin), F_GETFL);
            fcntl(fileno(stdin), F_SETFL, O_NONBLOCK);
            tcgetattr(STDIN_FILENO, &new_termios);
            new_termios.c_lflag &= ~(ICANON | ECHO);
            new_termios.c_cc[VMIN]  = 1;
            new_termios.c_cc[VTIME] = 0;
            tcsetattr(STDIN_FILENO, TCSAFLUSH, &new_termios);
            stdin_was_changed = 1;
        }
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
char ezfb_kb_hit()
{
    char c   ;
    int  ret ;
    if(!stdin_was_changed)
        ezfb_set_input_hot();
    ret = read(STDIN_FILENO, &c, 1);
#if EZFB_API_FUNCTION_NIT_PICKING
    if((ret != -1) && c)
        printf("ezfb_kb_hit : ret = %d : c = %d\n", ret, c);
#endif
    return (ret == -1) ? 0 : c;
}

//############################################################################
int ezfb_wait_4_key()
{
EZFB_FUNCTION_CALL_1
    while(!ezfb_kb_hit())
    {
        usleep(EZFB_WAIT_4_KEY_USECS);    
    }
EZFB_FUNCTION_RETURN(ret)
}

//############################################################################
int ezfb_get_key()
{
    char a = 0, b = 0, c;
    while(0 == (c = ezfb_kb_hit()))
    {
        usleep(EZFB_GET_KEY_USECS);
    }
    if(c == 0x1b)
    {
        b = ezfb_kb_hit();
        a = ezfb_kb_hit();
    }
#if EZFB_API_FUNCTION_NIT_PICKING
    printf("ezfb_get_key : %d : %d : %d\n", a, b, c);
#endif
    return (a << 16) | (b << 8) | c;
}

//############################################################################
int ezfb_get_dec_int()
{
    ezfb_restore_input();
    int  dec_int;
    scanf("%d", &dec_int);
    return dec_int;
}

//############################################################################
int ezfb_get_chars(char* line)
{    
    ezfb_restore_input();
    return scanf("%s", line);
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################

