//############################################################################
//
// ilda.cpp
//
// ~ International Laser Display Association ~
// ~     ILDA File Format Driver in C++      ~
// ~             Built for EZFB              ~
//
// by James Lehman
// james@akrobiz.com
//
// began: October 2003
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ilda.hpp"
    
//############################################################################
double pi         = (4 * atan(1.0)) ;
double two_pi     = (pi * 2.0)      ;
double half_pi    = (pi / 2.0)      ;
double one_degree = (pi / 180.0)    ;

//############################################################################
void print_ilda_error(int e)
{
    if(!e)
        cout << "    ILDA_OK" << endl;
    else
    {
        if(e & ILDA_BAD_HEADER           )  cout << "    ILDA_BAD_HEADER"           << endl;
        if(e & ILDA_EOF_ERROR            )  cout << "    ILDA_EOF_ERROR"            << endl;
        if(e & ILDA_FILE_OPEN_FAILED     )  cout << "    ILDA_FILE_OPEN_FAILED"     << endl;
        if(e & ILDA_UNKNOWN_FORMAT       )  cout << "    ILDA_UNKNOWN_FORMAT"       << endl;
        if(e & ILDA_INCOMPATIBLE_FORMATS )  cout << "    ILDA_INCOMPATIBLE_FORMATS" << endl;
    }
    return;
}

//############################################################################
// class ilda_section
//############################################################################
ilda_section& ilda_section::tell()
{
    cout << endl;
    switch(format)
    {
        case  0: cout << "    3D frame" << endl; break;
        case  1: cout << "    2D frame" << endl; break;
        case  2: cout << "    palette"  << endl; break;
        default: cout << "    unknown"  << endl; break;
    }
    cout << "    name     = " <<  name.c_str() << endl;
    cout << "    owner    = " << owner.c_str() << endl;
    cout << "    quantity = " <<      quantity << endl;
    cout << "    identity = " <<      identity << endl;
    cout << "    total    = " <<      total    << endl;
    cout << "    scanner  = " << (int)scanner  << endl;
    if(error)
        print_ilda_error(error);
    return *this;
}

//############################################################################
char ilda_section::get_point_color(u_int &point)
{
    if(format == 2) return 0;
    if(point <  0       ) point = quantity - 1;
    if(point >= quantity) point = 0;
    return image[point].c;
}

//############################################################################
ilda_section& ilda_section::set_point_color(u_int &point, u_char color)
{
    if(format == 0 || format == 1)
    {
        if(point <  0       ) point = quantity - 1;
        if(point >= quantity) point = 0;
        image[point].c = color;
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::set_points_color(u_int &point1, u_int &point2, u_char color)
{
    u_int i;
    if(format == 0 || format == 1)
    {
        if(point1 <  0       ) point1 = quantity - 1;
        if(point1 >= quantity) point1 = 0;
        if(point2 <  0       ) point2 = quantity - 1;
        if(point2 >= quantity) point2 = 0;
        for(i = point1; i != point2; i = (i++ % quantity))
            image[i].c = color;
        image[(i++ % quantity)].c = color;
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::blank_point(u_int &point)
{
    if(format == 0 || format == 1)
    {
        if(point <  0       ) point = quantity - 1;
        if(point >= quantity) point = 0;
        image[point].blank();
    }
    return *this;
}
//############################################################################
ilda_section& ilda_section::blank_points(u_int &point1, u_int &point2)
{
    u_int i;
    if(format == 0 || format == 1)
    {
        if(point1 <  0       ) point1 = quantity - 1;
        if(point1 >= quantity) point1 = 0;
        if(point2 <  0       ) point2 = quantity - 1;
        if(point2 >= quantity) point2 = 0;
        for(i = point1; i != point2; i = (i++ % quantity))
            image[i].blank();
        image[(i++ % quantity)].blank();
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::unblank_point(u_int &point)
{
    if(format == 0 || format == 1)
    {
        if(point <  0       ) point = quantity - 1;
        if(point >= quantity) point = 0;
        image[point].unblank();
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::unblank_points(u_int &point1, u_int &point2)
{
    if(format == 0 || format == 1)
    {
        u_int i;
        if(point1 <  0       ) point1 = quantity - 1;
        if(point1 >= quantity) point1 = 0;
        if(point2 <  0       ) point2 = quantity - 1;
        if(point2 >= quantity) point2 = 0;
        for(i = point1; i != point2; i = (i++ % quantity))
            image[i].unblank();
        image[(i++ % quantity)].unblank();
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::minimize_blank_points()
{
    if(format == 0 || format == 1)
    {
        vector<ilda_3D_point> minimized;

        for(u_int i = 0; i < quantity; i++)
        {
            if(!image[i].is_blank())
            {
                if(i)
                    minimized.push_back(image[i - 1]);
                while(!image[i].is_blank() && i < quantity)
                {
                    minimized.push_back(image[i]);
                    i++;
                }
                if(i < quantity)
                    minimized.push_back(image[i]);
            }
        }
        image = minimized;
        quantity = image.size();
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::flip(int plane)
{
    if(format == 0 || format == 1)
    {
        u_int i;
        switch(plane)
        {
           case 0: // X
                   for(i = 0; i < quantity; i++)
                       image[i].x = -image[i].x;
                   break;                                
           // --------------------------------------------------------------------
           case 1: // Y
                   for(i = 0; i < quantity; i++)
                       image[i].y = -image[i].y;
                   break;
           // --------------------------------------------------------------------
           case 3: // Z
                   for(i = 0; i < quantity; i++)
                       image[i].z = -image[i].z;
                   break;
           // --------------------------------------------------------------------
        }
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::quarter_turn(int plane, int turns)
{
    if(format == 0 || format == 1)
    {
        int i, j;
        short temp;
        switch(plane)
        {
           case 0: // X Y
                   for(i = 0; i < quantity; i++)
                       for(j = 0; j < turns; j++)
                       {
                           temp = -image[i].x;
                           image[i].x = image[i].y;
                           image[i].y = temp;
                       }
                   break;                                
           // --------------------------------------------------------------------
           case 1: // Z Y
                   for(i = 0; i < quantity; i++)
                       for(j = 0; j < turns; j++)
                       {
                           temp = -image[i].z;
                           image[i].z = image[i].y;
                           image[i].y = temp;
                       }
                   break;
           // --------------------------------------------------------------------
           case 2: // X Z
                   for(i = 0; i < quantity; i++)
                       for(j = 0; j < turns; j++)
                       {
                           temp = -image[i].x;
                           image[i].x = image[i].z;
                           image[i].z = temp;
                       }
                   break;
           // --------------------------------------------------------------------
        }
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::z_order_points(short int span)
{
    if(format == 0 || format == 1)
    {
        int step = span / quantity;
        for(u_int i = 0; i < quantity; i++)
            image[i].z = i * step - span / 2;
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::flatten_z()
{
    if(format == 0 || format == 1)
    {
        for(u_int i = 0; i < quantity; i++)
            image[i].z = 0;
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::concentric_sine_ripple(float amplitude, float freq, float phase)
{
    if(format == 0 || format == 1)
    {
        for(u_int i = 0; i < quantity; i++)
            image[i].z = short(amplitude * (16384.0 / pi) * sin(sqrt(float(image[i].x * image[i].x + image[i].y * image[i].y)) / 32768.0 * two_pi * freq + phase));
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::parallel_sine_ripple(int direction, float amplitude, float freq, float phase)
{
    if(format == 0 || format == 1)
    {
        u_int i;
        switch(direction)
        {
            case 0: for(i = 0; i < quantity; i++)
                        image[i].z = short(amplitude * (16384.0 / pi) * sin(image[i].x / 32768.0 * two_pi * freq + phase));
                    break;
            //----------------------------------------------------------------
            case 1: for(i = 0; i < quantity; i++)
                        image[i].z = short(amplitude * (16384.0 / pi) * sin(image[i].y / 32768.0 * two_pi * freq + phase));
                    break;
            //----------------------------------------------------------------
            case 2: for(i = 0; i < quantity; i++)
                        image[i].z = short(amplitude * (16384.0 / pi) * sin(image[i].x * image[i].y / 1073741824.0 * two_pi * freq + phase));
                    break;
            //----------------------------------------------------------------
        }
    }
    return *this;
}

//############################################################################
int ilda_section::move(short  dx, short  dy, short  dz)
{
    if(format == 2) return 0;
    u_int  i;
    for(i = 0; i < quantity; i++)
    {
        if((image[i].x + dx) > ILDA_MAX_X || (image[i].x + dx) < ILDA_MIN_X)
            return 1;
        //--------------------------------------------------------------------
        if((image[i].y + dy) > ILDA_MAX_Y || (image[i].y + dy) < ILDA_MIN_Y)
            return 2;
        //--------------------------------------------------------------------
        if((image[i].z + dz) > ILDA_MAX_Z || (image[i].z + dz) < ILDA_MIN_Z)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = 0; i < quantity; i++)
    {
        image[i].x += dx;
        image[i].y += dy;
        image[i].z += dz;
    }
    return 0;
}

//############################################################################
int ilda_section::scale(float  fx, float  fy, float  fz)
{
    if(format == 2) return 0;
    u_int i;
    for(i = 0; i < quantity; i++)
    {
        if((image[i].x * fx) > (float)ILDA_SHORT_MAX || (image[i].x * fx) < (float)ILDA_SHORT_MIN)
            return 1;
        //--------------------------------------------------------------------
        if((image[i].y * fy) > (float)ILDA_SHORT_MAX || (image[i].y * fy) < (float)ILDA_SHORT_MIN)
            return 2;
        //--------------------------------------------------------------------
        if((image[i].z * fz) > (float)ILDA_SHORT_MAX || (image[i].z * fy) < (float)ILDA_SHORT_MIN)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = 0; i < quantity; i++)
    {
        image[i].x = short(image[i].x * fx);
        image[i].y = short(image[i].y * fx);
        image[i].z = short(image[i].z * fx);
    }
    return 0;
}

//############################################################################
ilda_section& ilda_section::rainbow_recolor(int effect, u_char c1, u_char c2, float span)
{
    if(format == 0 || format == 1)
    {
        u_int  i, j, start, end;
        u_char color;
        switch(effect)
        {
            case 0: for(i = 0; i < quantity; i++) // span through x
                        image[i].c = u_char((image[i].x * span / 65536.0) * (c2 - c1)) % (c2 - c1) + c1;
                    break;
           //---------------------------------------------------------------------
            case 1: for(i = 0; i < quantity; i++) // span through y
                        image[i].c = u_char((image[i].y * span / 65536.0) * (c2 - c1)) % (c2 - c1) + c1;
                    break;
           //---------------------------------------------------------------------
            case 2: for(i = 0; i < quantity; i++) // span through z
                        image[i].c = u_char((image[i].z * span / 65536.0) * (c2 - c1)) % (c2 - c1) + c1;
                    break;
           //---------------------------------------------------------------------
            case 3: for(i = 0; i < quantity; i++) // span through points
                        image[i].c = u_char((i * span / float(quantity)) * (c2 - c1)) % (c2 - c1) + c1;
                    break;
           //---------------------------------------------------------------------
            case 4: for(i = 0; i < quantity; i++) // index palette through points
                        image[i].c = i % (c2 - c1) + c1;
                    break;
           //---------------------------------------------------------------------
            case 5: for(i = 1; i <= number_of_shapes(); i++) // index through shapes
                    {
                        color = ((i - 1) % (c2 - c1)) + c1;
                        find_shape(i, start, end);
                        for(j = start; j <= end; j++)
                            image[j].c = color;
                    }
                    break;
           //---------------------------------------------------------------------
            case 6: for(i = 1; i <= number_of_shapes(); i++) // span through shapes
                    {
                        color = u_char(((i - 1) * span / number_of_shapes()) * (c2 - c1)) % (c2 - c1) + c1;
                        find_shape(i, start, end);
                        for(j = start; j <= end; j++)
                            image[j].c = color;
                    }
                    break;
           //---------------------------------------------------------------------
        }
    }
    return *this;
}

//############################################################################
ilda_section& ilda_section::palette_cycle(u_char c1, u_char c2, int step)
{
    if(format == 0 || format == 1)
    {
        for(u_int i = 0; i < quantity; i++)
            image[i].c = ((image[i].c - c1 + step) % (c2 - c1)) + c1;
    }
    return *this;
}

//############################################################################
void ilda_section::rotate_point(short   x, short   y, short   z,
                                float& rx, float& ry, float& rz,
                                float  ax, float  ay, float  az)
{
    float sin_of_ax = sin(ax),
          sin_of_ay = sin(ay),
          sin_of_az = sin(az),
          cos_of_ax = cos(ax),
          cos_of_ay = cos(ay),
          cos_of_az = cos(az);
                    
    ry =  y * cos_of_ax -  z * sin_of_ax;
    rz =  z * cos_of_ax +  y * sin_of_ax; // rotate around x
    rz = rz * cos_of_ay -  x * sin_of_ay;
    rx =  x * cos_of_ay + (z * cos_of_ax +  y * sin_of_ax) * sin_of_ay; // rotate around y
    rx = rx * cos_of_az - (y * cos_of_ax -  z * sin_of_ax) * sin_of_az;
    ry = ry * cos_of_az + (x * cos_of_ay + (z * cos_of_ax +  y * sin_of_ax) * sin_of_ay) * sin_of_az; // rotate around z
    return;
}

//############################################################################
void ilda_section::rotate_point_around_point(short   x, short   y, short   z,
                                             short  cx, short  cy, short  cz,
                                             float& rx, float& ry, float& rz,
                                             float  ax, float  ay, float  az)
{
    float sin_of_ax = sin(ax),
          sin_of_ay = sin(ay),
          sin_of_az = sin(az),
          cos_of_ax = cos(ax),
          cos_of_ay = cos(ay),
          cos_of_az = cos(az);
    x -= cx;
    y -= cy;
    z -= cz;    
    ry =  y * cos_of_ax -  z * sin_of_ax;
    rz =  z * cos_of_ax +  y * sin_of_ax; // rotate around x
    rz = rz * cos_of_ay -  x * sin_of_ay;
    rx =  x * cos_of_ay + (z * cos_of_ax +  y * sin_of_ax) * sin_of_ay; // rotate around y
    rx = rx * cos_of_az - (y * cos_of_ax -  z * sin_of_ax) * sin_of_az;
    ry = ry * cos_of_az + (x * cos_of_ay + (z * cos_of_ax +  y * sin_of_ax) * sin_of_ay) * sin_of_az; // rotate around z
    rx += cx;
    ry += cy;
    rz += cz;
    return;
}

//############################################################################
void ilda_section::scale_point_around_point(short   x, short   y, short   z,
                                            short  cx, short  cy, short  cz,
                                            float& sx, float& sy, float& sz,
                                            float  mx, float  my, float  mz)
{
    x -= cx;
    y -= cy;
    z -= cz;    
    sx = x * mx;
    sy = y * my;
    sz = z * mz;
    sx += cx;
    sy += cy;
    sz += cz;
}

//############################################################################
ilda_section& ilda_section::rotate_all_around_origin(float  ax, float  ay, float  az)
{
    if(format == 0 || format == 1)
    {
        float rx, ry, rz;
        for(u_int i = 0; i < quantity; i++)
        {
            rotate_point(image[i].x, image[i].y, image[i].z, rx, ry, rz, ax, ay, az);
            image[i].x = short(rx);
            image[i].y = short(ry);
            image[i].z = short(rz);
        }
    }
    return *this;
}

//############################################################################
u_int ilda_section::number_of_shapes()
{
    if(format == 2) return 0;
    u_int i,
          in_shape = 0,
          shapes   = 0;

    for(i = 0; i < quantity; i++)
    {
          if(!image[i].is_blank() && !in_shape)
          {
              shapes++;
              in_shape = 1;
          }
          else if(image[i].is_blank())
              in_shape = 0;
    }
    return shapes;
}
    
//############################################################################
void ilda_section::find_shape(u_int shape, u_int& start, u_int& end)
{
    if(format == 2) return;
    u_int i,
          shape_num = 0;

    shape %= number_of_shapes();

    for(i = 0; i < quantity; i++)
    {
          if(!image[i].is_blank())
          {
              start = (i) ? (i - 1) : (0);
              while(!image[i].is_blank() && i < quantity)
                  i++;
              if(i < quantity)
                  end = i;
              else
                  end = quantity - 1;
              if(shape == shape_num)
                  return;
              shape_num++;
          }
    }
    return;
}

//############################################################################
void ilda_section::find_shape_center(u_int shape, short& cx, short& cy, short& cz)
{
    u_int i,
          start,
          end;
    int   max_x = -40000, // more negative than any short
          min_x =  40000, // more positive than any short
          max_y = -40000,
          min_y =  40000,
          max_z = -40000,
          min_z =  40000;
    find_shape(shape, start, end);
    for(i = start; i <= end; i++)
    {
        if(image[i].x >= max_x) max_x = image[i].x;
        if(image[i].x <= min_x) min_x = image[i].x;
        if(image[i].y >= max_y) max_y = image[i].y;
        if(image[i].y <= min_y) min_y = image[i].y;
        if(image[i].z >= max_z) max_z = image[i].z;
        if(image[i].z <= min_z) min_z = image[i].z;
    }
    cx = short(max_x - ((max_x - min_x) / 2));
    cy = short(max_y - ((max_y - min_y) / 2));
    cz = short(max_z - ((max_z - min_z) / 2));
    return;
}

//############################################################################
int ilda_section::move_shape(u_int shape, short dx, short dy, short dz)
{
    if(format == 2) return 0;
    u_int  i,
           start,
           end;
    find_shape(shape, start, end);
    for(i = start; i <= end; i++)
    {
        if((image[i].x + dx) > ILDA_MAX_X || (image[i].x + dx) < ILDA_MIN_X)
            return 1;
        //--------------------------------------------------------------------
        if((image[i].y + dy) > ILDA_MAX_Y || (image[i].y + dy) < ILDA_MIN_Y)
            return 2;
        //--------------------------------------------------------------------
        if((image[i].z + dz) > ILDA_MAX_Z || (image[i].z + dz) < ILDA_MIN_Z)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = start; i <= end; i++)
    {
        image[i].x += dx;
        image[i].y += dy;
        image[i].z += dz;
    }
    return 0;
}

//############################################################################
int ilda_section::rotate_shape(u_int shape, float ax, float ay, float az)
{
    if(format == 2) return 0;
    u_int  i,
           start,
           end;
    short  cx = 0,
           cy = 0,
           cz = 0;
    float  rx, ry, rz;

    find_shape(shape, start, end);
    find_shape_center(shape, cx, cy, cz);

    for(i = start; i <= end; i++)
    {
        rotate_point_around_point(image[i].x, image[i].y, image[i].z, cx, cy, cz, rx, ry, rz, ax, ay, az);
        //--------------------------------------------------------------------
        if(rx > ILDA_MAX_X || rx < ILDA_MIN_X)
            return 1;
        //--------------------------------------------------------------------
        if(ry > ILDA_MAX_Y || ry < ILDA_MIN_Y)
            return 2;
        //--------------------------------------------------------------------
        if(rz > ILDA_MAX_Z || rz < ILDA_MIN_Z)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = start; i <= end; i++)
    {
        rotate_point_around_point(image[i].x, image[i].y, image[i].z, cx, cy, cz, rx, ry, rz, ax, ay, az);
        image[i].x = short(rx);
        image[i].y = short(ry);
        image[i].z = short(rz);
    }
    return 0;
}

//############################################################################
int ilda_section::rotate_shape_around_origin(u_int shape, float ax, float ay, float az)
{
    if(format == 2) return 0;
    u_int  i,
           start,
           end;
    float  rx, ry, rz;
    find_shape(shape, start, end);
    for(i = start; i <= end; i++)
    {
        rotate_point(image[i].x, image[i].y, image[i].z, rx, ry, rz, ax, ay, az);
        //--------------------------------------------------------------------
        if(rx > ILDA_MAX_X || rx < ILDA_MIN_X)
            return 1;
        //--------------------------------------------------------------------
        if(ry > ILDA_MAX_Y || ry < ILDA_MIN_Y)
            return 2;
        //--------------------------------------------------------------------
        if(rz > ILDA_MAX_Z || rz < ILDA_MIN_Z)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = start; i <= end; i++)
    {
        rotate_point(image[i].x, image[i].y, image[i].z, rx, ry, rz, ax, ay, az);
        image[i].x = short(rx);
        image[i].y = short(ry);
        image[i].z = short(rz);
    }
    return 0;
}

//############################################################################
int ilda_section::scale_shape(u_int shape, float mx, float my, float mz)
{
    if(format == 2) return 0;
    u_int  i,
           start,
           end;
    short  cx = 0,
           cy = 0,
           cz = 0;
    float  sx, sy, sz;

    find_shape        (shape, start, end);
    find_shape_center (shape, cx, cy, cz);

    for(i = start; i <= end; i++)
    {
        scale_point_around_point(image[i].x, image[i].y, image[i].z, cx, cy, cz, sx, sy, sz, mx, my, mz);
        //--------------------------------------------------------------------
        if(sx > ILDA_MAX_X || sx < ILDA_MIN_X)
            return 1;
        //--------------------------------------------------------------------
        if(sy > ILDA_MAX_Y || sy < ILDA_MIN_Y)
            return 2;
        //--------------------------------------------------------------------
        if(sz > ILDA_MAX_Z || sz < ILDA_MIN_Z)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = start; i <= end; i++)
    {
        scale_point_around_point(image[i].x, image[i].y, image[i].z, cx, cy, cz, sx, sy, sz, mx, my, mz);
        image[i].x = short(sx);
        image[i].y = short(sy);
        image[i].z = short(sz);
    }
    return 0;
}

//############################################################################
int ilda_section::scale_shape_around_origin(u_int shape, float mx, float my, float mz)
{
    if(format == 2) return 0;
    u_int  i,
           start,
           end;
    float  sx, sy, sz;
    find_shape(shape, start, end);
    for(i = start; i <= end; i++)
    {
        sx = image[i].x * mx;
        if(sx > ILDA_MAX_X || sx < ILDA_MIN_X)
            return 1;
        //--------------------------------------------------------------------
        sy = image[i].y * my;
        if(sy > ILDA_MAX_Y || sy < ILDA_MIN_Y)
            return 2;
        //--------------------------------------------------------------------
        sz = image[i].z * mz;
        if(sz > ILDA_MAX_Z || sz < ILDA_MIN_Z)
            return 3;
        //--------------------------------------------------------------------
    }
    for(i = start; i <= end; i++)
    {
        image[i].x = short(image[i].x * mx);
        image[i].y = short(image[i].y * my);
        image[i].z = short(image[i].z * mz);
    }
    return 0;
}

//############################################################################
void ilda_section::wiggle_shapes(int plane, int max)
{
    if(format == 2) return;
    u_int  shape;
    int    dx, dy, dz;

    switch(plane)
    {
        case 0: // X Y
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        dx = int((ezfb_random_01() - 0.5) * 2.0 * max);
                        dy = int((ezfb_random_01() - 0.5) * 2.0 * max);                      
                    } while(move_shape(shape, dx, dy, 0)); // non zero is out of bounds
                }
                break;
        //--------------------------------------------------------------------
        case 1: // Z Y
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        dz = int((ezfb_random_01() - 0.5) * 2.0 * max);
                        dy = int((ezfb_random_01() - 0.5) * 2.0 * max);                      
                    } while(move_shape(shape, 0, dy, dz));
                }
                break;
        //--------------------------------------------------------------------
        case 2: // X Z
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        dx = int((ezfb_random_01() - 0.5) * 2.0 * max);
                        dz = int((ezfb_random_01() - 0.5) * 2.0 * max);                      
                    } while(move_shape(shape, dx, 0, dz));
                }
                break;
        //--------------------------------------------------------------------
        case 3: // X Y Z
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        dx = int((ezfb_random_01() - 0.5) * 2.0 * max);
                        dy = int((ezfb_random_01() - 0.5) * 2.0 * max);
                        dz = int((ezfb_random_01() - 0.5) * 2.0 * max);                      
                    } while(move_shape(shape, dx, dy, dz));
                }
                break;
        //--------------------------------------------------------------------
    }
    return;
}

//############################################################################
void ilda_section::roto_wiggle_shapes(int plane, float max)
{
    if(format == 2) return;
    u_int shape;
    float ax, ay, az;
    switch(plane)
    {
      case 0: // X Y
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        az = (ezfb_random_01() - 0.5) * 2.0 * max;
                    } while(rotate_shape(shape, 0, 0, az));
                }
                break;
      //----------------------------------------------------------------------
      case 1: // Z Y
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        ax = (ezfb_random_01() - 0.5) * 2.0 * max;
                    } while(rotate_shape(shape, ax, 0, 0));
                }
                break;
      //----------------------------------------------------------------------
      case 2: // X Z
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        ay = (ezfb_random_01() - 0.5) * 2.0 * max;
                    } while(rotate_shape(shape, 0, ay, 0));
                }
                break;
      //----------------------------------------------------------------------
      case 3: // X Y Z
                for(shape = 0; shape < number_of_shapes(); shape++)
                {                  
                    do
                    {
                        ax = (ezfb_random_01() - 0.5) * 2.0 * max;
                        ay = (ezfb_random_01() - 0.5) * 2.0 * max;
                        az = (ezfb_random_01() - 0.5) * 2.0 * max;
                    } while(rotate_shape(shape, ax, ay, az));
                }
                break;
      //----------------------------------------------------------------------
    }
    return;
}

//############################################################################
void ilda_section::falling_shapes(int plane, int max)
{
    if(format == 2) return;
    u_int i,
          shape,
          start,
          end;
    int   //dx,
          dy,
          //dz,
          //out_x = 0,
          //out_z = 0,
          out_y = 0;

    for(shape = 0; shape < number_of_shapes(); shape++)
        switch(plane)
        {
            case 0:
                     //dx = int((ezfb_random_01() - 0.5) * 2.0 * max);
                    dy = int((ezfb_random_01() + 1.0) * 2.0 * max); // all negative in Y
                    //dz = int((ezfb_random_01() - 0.5) * 2.0 * max);
                    find_shape(shape, start, end);
                    for(i = start; i <= end; i++)
                    {
                        //if((image[i].x + dx) > ILDA_MAX_X || (image[i].x + dx) < ILDA_MIN_X) out_x = 1;
                        if((image[i].y + dy) > ILDA_MAX_Y || (image[i].y + dy) < ILDA_MIN_Y) out_y = 1;
                        //if((image[i].z + dz) > ILDA_MAX_Z || (image[i].z + dz) < ILDA_MIN_Z) out_z = 1;
                    }
                    //if(out_x)
                    //    for(i = start; i <= end; i++)
                    //        image[i].x = -image[i].x;
                    if(out_y)
                        for(i = start; i <= end; i++)
                            image[i].y = -image[i].y;
                    //if(out_z)
                    //    for(i = start; i <= end; i++)
                    //        image[i].z = -image[i].z;
                    for(i = start; i <= end; i++)
                    {
                        //if(!out_x) image[i].x += dx;
                        if(!out_y) image[i].y += dy;
                        //if(!out_z) image[i].z += dz;
                    }
                    //out_x = 0;
                    out_y = 0;
                    //out_z = 0;
                    break;
        //--------------------------------------------------------------------
        }
    //roto_wiggle_shapes(3, pi/50.0);
    return;
}

//############################################################################
void ilda_section::from_ifstream(ifstream& in)
{
    u_int              i;
    char               p, q;
    ilda_3D_point      point ;
    ilda_palette_item  color ;

    name.erase();
    owner.erase();
    image.clear();
    palette.clear();
    error = ILDA_OK;

    if(!in.eof())
    {
        in.get(p); if(p != 'I') { error = ILDA_BAD_HEADER; return; }
        in.get(p); if(p != 'L') { error = ILDA_BAD_HEADER; return; }
        in.get(p); if(p != 'D') { error = ILDA_BAD_HEADER; return; }
        in.get(p); if(p != 'A') { error = ILDA_BAD_HEADER; return; }
        in.get(p); if(p !=  0 ) { error = ILDA_BAD_HEADER; return; }
        in.get(p); if(p !=  0 ) { error = ILDA_BAD_HEADER; return; }
        in.get(p); if(p !=  0 ) { error = ILDA_BAD_HEADER; return; }

        in.get(format);

        for(i = 0; i < 8; i++)
        {
            in.get(p);
            name += p;
        }

        for(i = 0; i < 8; i++)
        {
            in.get(p);
            owner += p;
        }

        in.get(p);
        in.get(q);
        quantity = ((p & 0x00ff) << 8 | (q & 0x00ff));

        in.get(p);
        in.get(q);
        identity = ((p & 0x00ff) << 8 | (q & 0x00ff));

        in.get(p);
        in.get(q);
        total    = ((p & 0x00ff) << 8 | (q & 0x00ff));

        in.get(scanner);
        in.get(future);

        if(quantity)  
        {
            switch(format)
            {
                case  0:
                         for(i = 0; i < quantity; i++)
                         {   
                             in.get(p);
                             in.get(q);
                             point.x = ((p & 0x00ff) << 8 | (q & 0x00ff));
                             in.get(p);
                             in.get(q);
                             point.y = ((p & 0x00ff) << 8 | (q & 0x00ff));
                             in.get(p);
                             in.get(q);
                             point.z = ((p & 0x00ff) << 8 | (q & 0x00ff));
                             in.get(point.k);
                             in.get(point.c);
                             image.push_back(point);
                         }         
                         break;
                //----------------------------------------------------------------
                case  1:
                         for(i = 0; i < quantity; i++)
                         {   
                             in.get(p);
                             in.get(q);
                             point.x = ((p & 0x00ff) << 8 | (q & 0x00ff));
                             in.get(p);
                             in.get(q);
                             point.y = ((p & 0x00ff) << 8 | (q & 0x00ff));
                             in.get(point.k);
                             in.get(point.c);
                             image.push_back(point);
                         }         
                         break;
                //----------------------------------------------------------------
                case  2: for(i = 0; i < quantity; i++)
                         {   
                             in.get(color.r);
                             in.get(color.g);
                             in.get(color.b);
                             palette.push_back(color);
                         }         
                         break;
                //----------------------------------------------------------------
            }       
        }
    }
    if(in.bad())
        error = ILDA_EOF_ERROR;
#if ILDA_DEBUG_MESSAGES
    tell();
#endif
    return;
}

//############################################################################
void ilda_section::to_ofstream(ofstream& out)
{
    u_int i;

    out.put('I');
    out.put('L');
    out.put('D');
    out.put('A');
    out.put( 0 );
    out.put( 0 );
    out.put( 0 );

    out.put(format);

    for(i = 0; i < 8; i++)
        out.put(name[i]);

    for(i = 0; i < 8; i++)
        out.put(owner[i]);

    out.put((char)((quantity & 0xff00) >> 8));
    out.put((char) (quantity & 0x00ff)      );

    out.put((char)((identity & 0xff00) >> 8));
    out.put((char) (identity & 0x00ff)      );

    out.put((char)((total    & 0xff00) >> 8));
    out.put((char) (total    & 0x00ff)      );

    out.put(scanner);

    out.put(0); // future

    switch(format)
    {
        case  0: for(i = 0; i < quantity; i++)
                 {   
                     out.put((char)((image[i].x & 0xff00) >> 8));
                     out.put((char) (image[i].x & 0x00ff)      );
                     out.put((char)((image[i].y & 0xff00) >> 8));
                     out.put((char) (image[i].y & 0x00ff)      );
                     out.put((char)((image[i].z & 0xff00) >> 8));
                     out.put((char) (image[i].z & 0x00ff)      );
                     out.put(image[i].k);
                     out.put(image[i].c);                     
                 }         
                 break;
            //----------------------------------------------------------------
        case  1: for(i = 0; i < quantity; i++)
                 {   
                     out.put((char)((image[i].x & 0xff00) >> 8));
                     out.put((char) (image[i].x & 0x00ff)      );
                     out.put((char)((image[i].y & 0xff00) >> 8));
                     out.put((char) (image[i].y & 0x00ff)      );
                     out.put(image[i].k);
                     out.put(image[i].c);                     
                 }         
                 break;
            //----------------------------------------------------------------
        case  2: for(i = 0; i < quantity; i++)
                 {   
                     out.put(palette[i].r);
                     out.put(palette[i].g);
                     out.put(palette[i].b);
                 }         
                 break;
            //----------------------------------------------------------------
    }
    return;
}

//############################################################################






//############################################################################
#ifdef __WITH_BMPTOOLS__
//############################################################################
void  ilda_section::render_as_bmp(int size, float ax, float ay, float az)
{
    if(quantity && ((format == 0) || (format == 1)))
    {
        char file_name[14];
        int   i;
        float rx1, ry1, rz1,
              rx2, ry2, rz2,
              scale = (bmp->yres / 65536.0);
        sprintf(file_name, "%08d.bmp", identity);
        BMP_8 bmp(size, size, file_name);
        bmp.palette();
        bmp.report();
        for(i = 1; i < quantity; i++)
        {
            if(!image[i].is_blank())
            {
                if(ax || ay || az)
                {
                    rotate_point(image[i - 1].x, image[i - 1].y, image[i - 1].z, rx1, ry1, rz1, ax, ay, az);
                    rotate_point(image[i    ].x, image[i    ].y, image[i    ].z, rx2, ry2, rz2, ax, ay, az);
                }
                else
                {
                    rx1 = image[i - 1].x;
                    ry1 = image[i - 1].y;
                    rz1 = image[i - 1].z;
                    rx2 = image[i    ].x;
                    ry2 = image[i    ].y;
                    rz2 = image[i    ].z;
                }
                bmp.line( (int)(rx1 * scale) + bmp.x / 2,
                          (int)(ry1 * scale) + bmp.x / 2,
                          (int)(rx2 * scale) + bmp.x / 2,
                          (int)(ry2 * scale) + bmp.x / 2,
                          image[i].c                            );

            }
        }
        bmp.save();
    }
    return;
}
//############################################################################
#endif
//############################################################################












//############################################################################
#ifdef __WITH_EZFB__
//############################################################################
ilda_section::ilda_section(ezfb_bitmap* bmp, short id)
                          : format    (2      ),
                            quantity  (256    ),
                            identity  (id     ),
                            total     (0      ),
                            scanner   (0      ),
                            future    (0      ),
                            error     (ILDA_OK)
{
    int i;
    ilda_palette_item color;
    for(i = 0; i < 256; i++)
    {
        color.r = ((bmp->palette[i] & 0x0000ff00) >> 8 );
        color.g = ((bmp->palette[i] & 0x00ff0000) >> 16);
        color.b = ((bmp->palette[i] & 0xff000000) >> 24);
        palette.push_back(color);
    }
}

//############################################################################
void ilda_section::render_in_ezfb(struct ezfb* fb, float ax, float ay, float az)
{
    if(quantity && ((format == 0) || (format == 1)))
    {
        int   i;
        float rx1, ry1, rz1,
              rx2, ry2, rz2,
              scale = (fb->Var.yres / 65536.0);
        for(i = 1; i < quantity; i++)
        {
            if(!image[i].is_blank())
            {
                if(ax || ay || az)
                {
                    rotate_point(image[i - 1].x, image[i - 1].y, image[i - 1].z, rx1, ry1, rz1, ax, ay, az);
                    rotate_point(image[i    ].x, image[i    ].y, image[i    ].z, rx2, ry2, rz2, ax, ay, az);
                }
                else
                {
                    rx1 = image[i - 1].x;
                    ry1 = image[i - 1].y;
                    rz1 = image[i - 1].z;
                    rx2 = image[i    ].x;
                    ry2 = image[i    ].y;
                    rz2 = image[i    ].z;
                }
                ezfb_put_line(fb,                 (int)(image[i - 1].x * scale) + fb->mid_y
                                , fb->Var.yres - ((int)(image[i - 1].y * scale) + fb->mid_y)
                                ,                 (int)(image[i    ].x * scale) + fb->mid_y
                                , fb->Var.yres - ((int)(image[i    ].y * scale) + fb->mid_y)
                                , image[i].c
                             );
            }
        }
    }
    else if(format == 2)
    {
        u_int i, j, k = 0, x, y;
        for(i = 0; i < 16; i++)
            for(j = 0; j < 16; j++)
            {
                for(y = 0; y < (fb->Var.yres / 16 - 1); y++)
                    for(x = 0; x < (fb->Var.yres / 16 - 1); x++)
                        ezfb_put_pixel(fb, j * (fb->Var.xres / 16) + x, i * (fb->Var.yres / 24) + y, k);
                k++;
            }
    }
}

//############################################################################
void ilda_section::render_in_ezfb_bitmap(struct ezfb_bitmap* bmp, float ax, float ay, float az)
{
    bmp_clear(bmp, bmp->black);
    if(quantity && ((format == 0) || (format == 1)))
    {
        int   i;
        float rx1, ry1, rz1,
              rx2, ry2, rz2,
              scale = (bmp->yres / 65536.0);
        bmp_put_rectangle(bmp, 0, 0, bmp->yres - 1, bmp->yres - 1, bmp->white, 0);
        for(i = 1; i < quantity; i++)
        {
            if(!image[i].is_blank())
            {
                if(ax || ay || az)
                {
                    rotate_point(image[i - 1].x, image[i - 1].y, image[i - 1].z, rx1, ry1, rz1, ax, ay, az);
                    rotate_point(image[i    ].x, image[i    ].y, image[i    ].z, rx2, ry2, rz2, ax, ay, az);
                }
                else
                {
                    rx1 = image[i - 1].x;
                    ry1 = image[i - 1].y;
                    rz1 = image[i - 1].z;
                    rx2 = image[i    ].x;
                    ry2 = image[i    ].y;
                    rz2 = image[i    ].z;
                }
                bmp_put_line(bmp,              (int)(rx1 * scale) + bmp->yres / 2
                                , bmp->yres - ((int)(ry1 * scale) + bmp->yres / 2)
                                ,              (int)(rx2 * scale) + bmp->yres / 2
                                , bmp->yres - ((int)(ry2 * scale) + bmp->yres / 2)
                                , image[i].c
                            );
            }
        }
    }
    else if(format == 2)
    {
        u_int i, j, k = 0, x, y;
        for(i = 0; i < 16; i++)
            for(j = 0; j < 16; j++)
            {
                for(y = 0; y < (bmp->yres / 16 - 1); y++)
                    for(x = 0; x < (bmp->yres / 16 - 1); x++)
                        bmp_put_pixel(bmp, j * (bmp->yres / 16) + x, i * (bmp->yres / 16) + y, k);
                k++;
                cout << '.';
            }
    }
    int factor = (bmp->yres / 384);
    bmp_printf(bmp, bmp->yres + 10, 256 * factor + 8 + 45 * factor, bmp->white, bmp->black, 0, factor, "%d", identity);
    return;
}
    
//############################################################################
void ilda_section::render_in_ezfb_bitmap_point_select(  ezfb_bitmap* bmp,
                                                        u_int&       point1,
                                                        u_int&       point2,
                                                        float        ax,
                                                        float        ay,
                                                        float        az
                                                     )
{
    if(quantity && ((format == 0) || (format == 1)))
    {
        float rx1, ry1, rz1, rx2, ry2, rz2,
              scale = (bmp->yres / 65536.0);
        
        if(point1 <  0       ) point1 = quantity - 1;
        if(point1 >= quantity) point1 = 0;
        if(point2 <  0       ) point2 = quantity - 1;
        if(point2 >= quantity) point2 = 0;
                
        render_in_ezfb_bitmap(bmp, ax, ay, az);

        if(ax || ay || az)
        {
            rotate_point(image[point1].x, image[point1].y, image[point1].z, rx1, ry1, rz1, ax, ay, az);
            rotate_point(image[point2].x, image[point2].y, image[point2].z, rx2, ry2, rz2, ax, ay, az);
        }
        else
        {
            rx1 = image[point1].x;
            ry1 = image[point1].y;
            rz1 = image[point1].z;
            rx2 = image[point2].x;
            ry2 = image[point2].y;
            rz2 = image[point2].z;
        }
        //--------------------------------------------------------------------
        // start point select cursor
        bmp_put_rectangle(bmp,         (int)((rx1 - 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry1 - 1000) * scale) + bmp->yres / 2),
                                       (int)((rx1 + 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry1 + 1000) * scale) + bmp->yres / 2),
                          (image[point1].is_blank()) ? (bmp->white) : (image[point1].c), 0
                         );
        //--------------------------------------------------------------------
        // end point select cursor
        bmp_put_line     (bmp,         (int)((rx2       ) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2 - 1000) * scale) + bmp->yres / 2),
                                       (int)((rx2       ) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2 + 1000) * scale) + bmp->yres / 2),
                          (image[point2].is_blank()) ? (bmp->white) : (image[point2].c)
                         );
        bmp_put_line     (bmp,         (int)((rx2 + 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2       ) * scale) + bmp->yres / 2),
                                       (int)((rx2 - 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2       ) * scale) + bmp->yres / 2),
                          (image[point2].is_blank()) ? (bmp->white) : (image[point2].c)
                         );
        bmp_put_line     (bmp,         (int)((rx2 - 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2 - 1000) * scale) + bmp->yres / 2),
                                       (int)((rx2 + 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2 + 1000) * scale) + bmp->yres / 2),
                          (image[point2].is_blank()) ? (bmp->white) : (image[point2].c)
                         );
        bmp_put_line     (bmp,         (int)((rx2 + 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2 - 1000) * scale) + bmp->yres / 2),
                                       (int)((rx2 - 1000) * scale) + bmp->yres / 2,
                          bmp->yres - ((int)((ry2 + 1000) * scale) + bmp->yres / 2),
                          (image[point2].is_blank()) ? (bmp->white) : (image[point2].c)
                         );
        //--------------------------------------------------------------------
    }
    return;
}

//############################################################################
void ilda_section::render_palette_in_ezfb_bitmap(ezfb_bitmap* bmp, u_char selected_index)
{
    int   i, j;
    int   factor = (bmp->yres / 384);

    for(i = 0; i < 256; i++)
        for(j = 0; j < factor; j++)
            bmp_put_line(bmp, bmp->yres, i * factor + j, bmp->yres + 50, i * factor + j, i);

    bmp_put_rectangle(bmp, bmp->yres + 70, 0, bmp->yres + 120, 256 * factor, selected_index, 1);

    for(j = 0; j < factor; j++)
        bmp_put_line(bmp, bmp->yres + 40, selected_index * factor + j, bmp->yres + 80, selected_index * factor + j, bmp->white);

    bmp_printf(bmp, bmp->yres + 10, 256 * factor + 8              , bmp->white, bmp->black, 0, factor, "i : %d", selected_index);
    bmp_printf(bmp, bmp->yres + 10, 256 * factor + 8 +  9 * factor, bmp->white, bmp->black, 0, factor, "r : %d", ((bmp->palette[selected_index] & 0x0000ff00) >>  8));
    bmp_printf(bmp, bmp->yres + 10, 256 * factor + 8 + 18 * factor, bmp->white, bmp->black, 0, factor, "g : %d", ((bmp->palette[selected_index] & 0x00ff0000) >> 16));
    bmp_printf(bmp, bmp->yres + 10, 256 * factor + 8 + 27 * factor, bmp->white, bmp->black, 0, factor, "b : %d", ((bmp->palette[selected_index] & 0xff000000) >> 24));
}

//############################################################################
#endif



//############################################################################
void ilda_section::render_raw(ofstream& out)
{
    if(quantity && ((format == 0) || (format == 1)))
    {
        int i, j, k = (int)ceil((float)9600.0 / quantity);

        for(j = 0; j < k; j++)
            for(i = 0; i < quantity; i++)
            {
                out.put( image[i].x & 0x00ff      );
                out.put((image[i].x & 0xff00) >> 8);
                out.put( image[i].y & 0x00ff      );
                out.put((image[i].y & 0xff00) >> 8);
            }
    }
}

//############################################################################
// class ilda
//############################################################################
int ilda::from_file(string file)
{
    int error = ILDA_OK;
    ifstream in(file.c_str(), ios::in | ios::binary);
    if(in.is_open())
    {
        file_name = file;
        error = from_ifstream(in);
        in.close();
    }
    else
    {
        cout << endl << file << " failed to open for input" << endl;
        error = ILDA_FILE_OPEN_FAILED;
    }
    return error;
}

//############################################################################
int ilda::from_ifstream(ifstream& in)
{
    int error = ILDA_OK;
    bool some_data_accepted = 0;
    ilda_section section;
    sections.clear();
    num_sections  = 0;
    num_frames    = 0;
    num_2D_frames = 0;
    num_3D_frames = 0;
    num_palettes  = 0;
    while(!in.eof() && !error)
    {
        section.from_ifstream(in);
        if(section.quantity && !section.error)
        {
            append_section(section);
            some_data_accepted = true;
            error |= section.error;
        }
    }
    if(error && some_data_accepted)
    {
        print_ilda_error(error);
        cout << "an error occurred" << endl;
        error = 0;
    }
    tell();
    return error;
} 

//############################################################################
void ilda::append_section(ilda_section section)
{
    sections.push_back(section);
    num_sections++;
    switch(section.format)
    {
        case  0:
                 num_frames++;
                 num_3D_frames++;
                 section.identity = num_frames;
                 break;

        case  1:
                 num_frames++;
                 num_2D_frames++;
                 section.identity = num_frames;
                 break;

        case  2: 
                 num_palettes++;
                 break;
    }
    return;
}

//############################################################################
void ilda::renumber_frames()
{
    u_int i, j = 0, k = 0;
    char number[7];
    for(i = 0; i < num_sections; i++)
    {
        switch(sections[i].format)
        {
            case  0:
            case  1:
                     sections[i].identity = j++;
                     sections[i].total = num_frames;
                     break;

            case  2: 
                     sections[i].identity = k++;
                     sections[i].total = num_palettes;
                     break;
        }
        sprintf(number, "%06d", sections[i].identity);
        sections[i].name  = "ES" + string(number);
        sections[i].owner = "LaserBoy";
    }
    return;
}

//############################################################################
void ilda::reverse()
{
    vector<ilda_section>           reverse;
    vector<ilda_section>::iterator palette_position = reverse.begin();
    for(u_int i = number_of_sections() - 1; ((int)i >= 0) ; i--)
        switch(sections[i].format)
        {
            case  0:
            case  1:
                     reverse.push_back(sections[i]);
                     break;
            //----------------------------------------------------------------
            case  2: reverse.insert(palette_position, sections[i]);
                     palette_position = reverse.end();
                     break;
            //----------------------------------------------------------------
        }        
    sections = reverse;
    renumber_frames();
    return;
}

//############################################################################
void ilda::save_as(string file)
{
    ofstream out(file.c_str(), ios::out | ios::binary);
    if(out.is_open())
    {
        to_ofstream(out);
        out.close();
    }
    else
        cout << endl << file << " failed to open for output" << endl;
}

//############################################################################
void ilda::to_ofstream(ofstream& out)
{
    u_int i;
    renumber_frames();
    for(i = 0; i < num_sections; i++)
    {
        sections[i].minimize_blank_points();
        if(sections[i].quantity)
            sections[i].to_ofstream(out);
    }
    (ilda_section(0, "End", "LaserBoy", 0, i, 0, 0)).to_ofstream(out);
    return;
}

//############################################################################
void ilda::recount()
{
    num_sections = number_of_sections();
    for(u_int i = 0; i < num_sections; i++)
    {
        switch(sections[i].format)
        {
            case  0: num_frames++;
                     num_3D_frames++;
                     break;
            //-------------------------
            case  1: num_frames++;
                     num_2D_frames++;
                     break;
            //-------------------------
            case  2: num_palettes++;
                     break;
            //-------------------------
        }
    }
    return;
}

//############################################################################
void ilda::tell()
{
    cout << "\n\n    total sections  = ";
    cout << num_sections;
    cout << "";
    cout << "\n    total frames    = ";
    cout << num_frames;
    cout << "";
    cout << "\n    total 2D frames = ";
    cout << num_2D_frames;
    cout << "";
    cout << "\n    total 3D frames = ";
    cout << num_3D_frames;
    cout << "";
    cout << "\n    total palettes  = ";
    cout << num_palettes << endl;
    return;
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################
