//########################################################################################
//
// ilda.hpp
//
// ~ International Laser Display Association ~
// ~     ILDA File Format Driver in C++      ~
// ~             Built for EZFB              ~
//
// by James Lehman
// james@akrobiz.com
//
// began: October 2003
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//########################################################################################

#ifndef __ILDA_DEFINITIONS__
#define __ILDA_DEFINITIONS__
    
//########################################################################################

#ifdef WIN32
 #define __WRITTEN_FOR_MICROSOFT__
  #pragma warning(disable: 4786)
  #pragma warning(disable: 4788)
  #include <limits>
#else
  #define __WRITTEN_FOR_LINUX__
#endif

//########################################################################################
#include <fstream>
#include <iterator>
#include <vector>
#include <string>
#include <iostream>

using namespace std;

//###################################################################################
//#define __WITH_BMPTOOLS__
#define __WITH_EZFB__

//#ifdef __WITH_BMPTOOLS__
//#include "bmp.hpp"
//#endif

#ifdef __WITH_EZFB__
#include "ezfb.h"
#include "ezfb_laser_palettes.h"
#endif

//###################################################################################
#pragma pack(1) // packs data structures down to 1 byte boundaries

//###################################################################################
#define ILDA_SHORT_MAX   32767
#define ILDA_SHORT_MIN  -32768
//-----------------------------------------------------------------------------------
// full scale cube
// used for bounds checking by any function that makes or moves points
#define ILDA_MAX_X       32767
#define ILDA_MIN_X      -32768
#define ILDA_MAX_Y       32767
#define ILDA_MIN_Y      -32768
#define ILDA_MAX_Z       32767
#define ILDA_MIN_Z      -32768
//-----------------------------------------------------------------------------------
// full scale spherical
// anything drawn inside this cube can be rotated to any
// angle and all points will be inside the full scale cube
//#define ILDA_MAX_X       23169
//#define ILDA_MIN_X      -23169
//#define ILDA_MAX_Y       23169
//#define ILDA_MIN_Y      -23169
//#define ILDA_MAX_Z       23169
//#define ILDA_MIN_Z      -23169
//-----------------------------------------------------------------------------------
// 7/8 cube
// slightly reduced limits for other applications that apply Z depth distortion
//#define ILDA_MAX_X       28672
//#define ILDA_MIN_X      -28672
//#define ILDA_MAX_Y       28672
//#define ILDA_MIN_Y      -28672
//#define ILDA_MAX_Z       28672
//#define ILDA_MIN_Z      -28672
//-----------------------------------------------------------------------------------
// 7/8 spherical
//#define ILDA_MAX_X       20274
//#define ILDA_MIN_X      -20274
//#define ILDA_MAX_Y       20274
//#define ILDA_MIN_Y      -20274
//#define ILDA_MAX_Z       20274
//#define ILDA_MIN_Z      -20274

//###################################################################################
#define    ILDA_OK                     0
#define    ILDA_BAD_HEADER             1
#define    ILDA_EOF_ERROR              2
#define    ILDA_FILE_OPEN_FAILED       4
#define    ILDA_UNKNOWN_FORMAT         8
#define    ILDA_INCOMPATIBLE_FORMATS  16
//-----------------------------------------------------------------------------------
#define    ILDA_DEBUG_MESSAGES         1
#define    ILDA_BLANKING_BIT          (0x01 << 6) // 01000000 = 64

//###################################################################################
class ilda_3D_point
{
public:
    ilda_3D_point(   short _x = 0
                   , short _y = 0
                   , short _z = 0
                   , char  _k = 0
                   , char  _c = 0
                 )
                 :   x(_x)
                   , y(_y)
                   , z(_z)
                   , k(_k)
                   , c(_c)
                                {};
    //-------------------------------------------------------------------------------
    ilda_3D_point(const ilda_3D_point& p)
                 : x(p.x), y(p.y), z(p.z), k(p.k), c(p.c)       {};
    //-------------------------------------------------------------------------------
    ilda_3D_point& operator = (const ilda_3D_point& p)
                                {
                                    x = p.x;
                                    y = p.y;
                                    z = p.z;
                                    k = p.k;
                                    c = p.c;
                                    return *this;
                                };
    //-------------------------------------------------------------------------------
    bool operator == (const ilda_3D_point& p)
                        {   
                            return(    x == p.x
                                    && y == p.y
                                    && z == p.z
                                    && k == p.k
                                    && c == p.c
                                   );
                        };
    //-------------------------------------------------------------------------------
    bool  is_blank()   {    return (k & ILDA_BLANKING_BIT);      };
    void  blank   ()   {    k |=  ILDA_BLANKING_BIT;   return;   };
    void  unblank ()   {    k &= ~ILDA_BLANKING_BIT;   return;   };
    //-------------------------------------------------------------------------------
    short x, y, z;
    char  k, c;
};

//###################################################################################
class ilda_palette_item
{
public:
    ilda_palette_item(char _r = 0, char _g = 0, char _b = 0)
                     : r(_r), g(_g), b(_b)                      {};
    //-------------------------------------------------------------------------------
    ilda_palette_item(const ilda_palette_item& c)
                     : r(c.r), g(c.g), b(c.b)                   {};
    //-------------------------------------------------------------------------------
    ilda_palette_item& operator = (const ilda_palette_item& c)
                                    {
                                        r = c.r;
                                        g = c.g;
                                        b = c.b;
                                        return *this;
                                    };
    //-------------------------------------------------------------------------------
    bool operator == (const ilda_palette_item& c)
                        {   return(r == c.r && g == c.g && b == c.b);    };
    //-------------------------------------------------------------------------------
    char r, g, b;
};

//###################################################################################
class ilda_section
{
public:
    //-------------------------------------------------------------------------------
    ilda_section()
                : format    (0      ),
                  quantity  (0      ),
                  identity  (0      ),
                  total     (0      ),
                  scanner   (0      ),
                  future    (0      ),
                  image     (       ),
                  palette   (       ),
                  error     (ILDA_OK)
                                                  {};
    //-------------------------------------------------------------------------------
    ilda_section(const ilda_section& section)
                : format    (section.format  ),
                  quantity  (section.quantity),
                  identity  (section.identity),
                  total     (section.total   ),
                  scanner   (section.scanner ),
                  future    (0               ),
                  image     (section.image   ),
                  palette   (section.palette ),
                  error     (ILDA_OK         )
                                                  {};
    //-------------------------------------------------------------------------------
    ilda_section(ilda_3D_point(*F)(int, int), int points, int frame)
                : format    (0      ),
                  quantity  (points ),
                  identity  (0      ),
                  total     (0      ),
                  scanner   (0      ),
                  future    (0      ),
                  image     (points ),
                  palette   (       ),
                  error     (ILDA_OK)
                                        {
                                            for(u_int i = 0; i < quantity; i++)
                                                image.push_back(F(i, frame));
                                        };
    //-------------------------------------------------------------------------------
    ilda_section(char f, string n, string o, short q, short i, short t, char s)
                : format    (f      ),
                  name      (n      ),
                  owner     (o      ),
                  quantity  (q      ),
                  identity  (i      ),
                  total     (t      ),
                  scanner   (s      ),
                  future    (0      ),
                  image     (       ),
                  palette   (       ),
                  error     (ILDA_OK)
                                        {};
    //-------------------------------------------------------------------------------
    ilda_section(ifstream& in)          { from_ifstream(in); };
    //-------------------------------------------------------------------------------
   ~ilda_section()                      {};
    //-------------------------------------------------------------------------------
    ilda_section& operator =                 (const ilda_section& section)
                                                {
                                                    format    = section.format   ;
                                                    quantity  = section.quantity ;
                                                    identity  = section.identity ;
                                                    total     = section.total    ;
                                                    scanner   = section.scanner  ;
                                                    future    = 0                ;
                                                    image     = section.image    ;
                                                    palette   = section.palette  ;
                                                    error     = ILDA_OK          ;
                                                    return *this;
                                                };
    //-------------------------------------------------------------------------------
    ilda_section& operator +=                (const ilda_section& section)
                                                {
                                                    if(format == 2 || section.format == 2)
                                                    {
                                                        error = ILDA_INCOMPATIBLE_FORMATS;
                                                        return *this;
                                                    }
                                                    image.push_back(image[image.size() - 1]); // last point in *this
                                                    image[image.size() - 1].blank();
                                                    image.push_back(section.image[0]); // first point in section
                                                    image[image.size() - 1].blank();
                                                    image.insert(image.end(), section.image.begin(), section.image.end());
                                                    quantity = image.size();
                                                    return *this;
                                                };
    //-------------------------------------------------------------------------------
    ilda_section  operator +                 (const ilda_section& section)
                                                {
                                                    ilda_section sum(*this);
                                                    sum += section;
                                                    return sum;
                                                };
    //-------------------------------------------------------------------------------
    ilda_section& tell                        ();
    char          get_point_color             (u_int &point);
    ilda_section& set_point_color             (u_int &point, u_char color);
    ilda_section& set_points_color            (u_int &point1, u_int &point2, u_char color);
    ilda_section& blank_point                 (u_int &point);
    ilda_section& blank_points                (u_int &point1, u_int &point2);
    ilda_section& unblank_point               (u_int &point);
    ilda_section& unblank_points              (u_int &point1, u_int &point2);
    ilda_section& minimize_blank_points       ();
    ilda_section& flip                        (int plane);
    ilda_section& quarter_turn                (int plane, int turns);
    
    ilda_section& z_order_points              (short span);
    ilda_section& flatten_z                   ();
    ilda_section& concentric_sine_ripple      (float amplitude, float freq, float phase);
    ilda_section& parallel_sine_ripple        (int direction, float amplitude, float freq, float phase);

    int           move                        (short  dx, short  dy, short  dz);
    int           scale                       (float  fx, float  fy, float  fz);
    ilda_section& rainbow_recolor             (int effect, u_char c1, u_char c2, float span);
    ilda_section& palette_cycle               (u_char c1, u_char c2, int step);
    //-------------------------------------------------------------------------------
    void          rotate_point                (short   x, short   y, short   z,
                                               float& rx, float& ry, float& rz,
                                               float  ax, float  ay, float  az);
    //-------------------------------------------------------------------------------
    void          rotate_point_around_point   (short   x, short   y, short   z,
                                               short  cx, short  cy, short  cz,
                                               float& rx, float& ry, float& rz,
                                               float  ax, float  ay, float  az);
    //-------------------------------------------------------------------------------
    void          scale_point_around_point    (short   x, short   y, short   z,
                                               short  cx, short  cy, short  cz,
                                               float& sx, float& sy, float& sz,
                                               float  mx, float  my, float  mz);
    //-------------------------------------------------------------------------------
    ilda_section& rotate_all_around_origin    (float  ax, float  ay, float  az);
    u_int         number_of_shapes            ();
    void          find_shape                  (u_int shape, u_int& start, u_int& end);
    void          find_shape_center           (u_int shape, short& cx, short& cy, short& cz);
    int           move_shape                  (u_int shape, short  dx, short  dy, short  dz);

    int           rotate_shape                (u_int shape, float ax, float ay, float az);
    int           rotate_shape_around_origin  (u_int shape, float ax, float ay, float az);

    int           scale_shape                 (u_int shape, float mx, float my, float mz);
    int           scale_shape_around_origin   (u_int shape, float mx, float my, float mz);

    void          wiggle_shapes               (int plane, int   max);
    void          roto_wiggle_shapes          (int plane, float max);
    void          falling_shapes              (int plane, int   max);

    void          from_ifstream               (ifstream& in );
    void          to_ofstream                 (ofstream& out);
    void          render_raw                  (ofstream& out);
    //-------------------------------------------------------------------------------
#ifdef __WITH_BMPTOOLS__
    void  render_as_bmp                      (int size, float ax = 0, float ay = 0, float az = 0);
#endif
    //-------------------------------------------------------------------------------
#ifdef __WITH_EZFB__
    //-------------------------------------------------------------------------------
    ilda_section                             (ezfb_bitmap* bmp, short id); // palette ctor
    //-------------------------------------------------------------------------------
    void  render_in_ezfb                     (ezfb*        fb , float ax = 0, float ay = 0, float az = 0);
    void  render_in_ezfb_bitmap              (ezfb_bitmap* bmp, float ax = 0, float ay = 0, float az = 0);
    //-------------------------------------------------------------------------------
    void  render_in_ezfb_bitmap_point_select (ezfb_bitmap* bmp, u_int &point1, u_int &point2,
                                                                float ax = 0, float ay = 0, float az = 0);
    //-------------------------------------------------------------------------------
    void  render_palette_in_ezfb_bitmap      (ezfb_bitmap* bmp, u_char selected_index);
#endif
    //-------------------------------------------------------------------------------
    char                       format;
    string                     name, owner;
    u_short                    quantity, identity, total;
    char                       scanner, future;
    vector<ilda_3D_point>      image;
    vector<ilda_palette_item>  palette;
    int                        error;
    //-------------------------------------------------------------------------------
};

//###################################################################################
class ilda
{
public:
    ilda()
        : file_name     ( ),
          num_sections  (0),
          num_frames    (0),
          num_2D_frames (0),
          num_3D_frames (0),
          num_palettes  (0),
          dud           ( )
                                 {};
    //-------------------------------------------------------------------------------
    ilda(const ilda& ild)
        : file_name     (ild.file_name    ),
          num_sections  (ild.num_sections ),
          num_frames    (ild.num_frames   ),
          num_2D_frames (ild.num_2D_frames),
          num_3D_frames (ild.num_3D_frames),
          num_palettes  (ild.num_palettes ),
          sections      (ild.sections     ),
          dud           (                 )
                                 {};
    //-------------------------------------------------------------------------------
    ilda(char* f)
        : num_sections  (0),
          num_frames    (0),
          num_2D_frames (0),
          num_3D_frames (0),
          num_palettes  (0),
          dud           ( )
                                 { from_file(f); };
    //-------------------------------------------------------------------------------
    ~ilda()                      {};
    //-------------------------------------------------------------------------------
    ilda& operator +=            (const ilda& ild     )
                                    {
                                        for(u_int i = 0; i < ild.num_sections; i++)
                                            append_section(ild.sections[i]);
                                        return *this;
                                    };
    //-------------------------------------------------------------------------------
    ilda  operator +             (const ilda& ild     )
                                    {
                                        ilda sum(*this);
                                        for(u_int i = 0; i < ild.num_sections; i++)
                                            sum.append_section(ild.sections[i]);
                                        return sum;
                                    };
    //-------------------------------------------------------------------------------
    ilda_section& operator []    (u_int index         )
                                    {
                                        if(index >= 0 && index < num_sections)
                                            return sections[index];
    
                                        cout << "\nERROR! index " << index << " is out of bounds";
                                        return dud;
                                    };
    //-------------------------------------------------------------------------------
    ilda_section& last_section   ()
                                    {
                                        return sections[number_of_sections() - 1];
                                    };
    //-------------------------------------------------------------------------------
    int         from_file        (string file         );
    int         from_ifstream    (ifstream& in        );

    void        append_section   (ilda_section section);
    void        append_section   () {    append_section(dud);    };
    void        renumber_frames  ();
    void        reverse          ();
    void        save_as          (string file         );
    void        to_ofstream      (ofstream& out       );
    void        recount          ();
    void        tell             ();

    //-------------------------------------------------------------------------------
    u_int    number_of_sections  () {  return sections.size(); };
    //-------------------------------------------------------------------------------
    string                file_name     ;
    u_int                 num_sections  ;
    u_int                 num_frames    ;
    u_int                 num_2D_frames ;
    u_int                 num_3D_frames ;
    u_int                 num_palettes  ;
    vector<ilda_section>  sections      ;
    ilda_section          dud           ;
};

//###################################################################################    
#pragma pack()

extern double pi;
extern double two_pi;
extern double half_pi;
extern double one_degree;

#endif

//###################################################################################
/////////////////////////////////////////////////////////////////////////////////////
//###################################################################################

