//############################################################################
//
// laserboy.cpp  (formerly ild2fb.cpp)
//
// ~   International Laser Display Association    ~
// ~ ILDA File Display and transformations in C++ ~
// ~               Built for EZFB                 ~
//
// by James Lehman
// james@akrobiz.com
//
// began : October 2003
// name changed to laserboy : January 2005
//
// This software is distributed in the hope that it will be useful, but
// without any warranty; without even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public License
// for more details.
//
//############################################################################

#include "ilda.hpp"

//############################################################################

#define FRAMES_SHARE (string("./frames/"))

//############################################################################
struct ezfb        fb  = {0}; // this is the frame buffer
struct ezfb_bitmap bmp = {0}; // this is a memory object same size as fb

//############################################################################
ilda_3D_point A(int point, int frame) // single frame, 40 points, 3D spring
{
    ilda_3D_point P;
    float factor = (point / 8.0) * two_pi;
    P.x = (short int)(900 * sin(factor));
    P.y = (short int)(900 * cos(factor));
    P.z = (short int)(point * 300 - 6000);
    P.k = 0;
    P.c = 1;
    return P;
}

//############################################################################
ilda_3D_point B(int point, int frame)
{
    ilda_3D_point P;
    float factor = (point / 20.0) * two_pi,
          shift  = (two_pi / 60.0) * frame;
    P.x = (short int)(point * 1200);
    P.y = (short int)(sin(3.5 * factor + shift) * point * 200);
    P.z = (short int)(0);
    P.k = 0;
    P.c = point * 2 + 1;
    return P;
}

//############################################################################
// The functions A, B and F are definitions for all points in a frame
// Calling the ctor ilda_section(ilda_3D_point(*F)(int, int), int points, int frame);
// is where the total number of points is defined
//----------------------------------------------------------------------------
ilda_3D_point F(int point, int frame)
{
    ilda_3D_point P;
    float factor = (point / 699.0) * 30 * pi;
    P.x = (short int)(22000 * sin(factor) * (sin(31*factor/15)) );
    P.y = (short int)(22000 * cos(factor) * (sin(31*factor/15)) );
    P.z = (short int)((point * 40) - 14000);
    P.k = 0;
    P.c = point % 63;
    return P;
}

//############################################################################
void show_hot_keys()
{
    cout << endl << endl <<
    "F1_KEY:      display this list of hot keys\n"
    "F2_KEY:      open another ILDA file in " << FRAMES_SHARE << endl << 
    "F3_KEY:      toggle point select cursor visibility\n"
    "F4_KEY:      cycle global palette\n"
    "ARROW_UP:    display next frame, press and hold for animations\n"
    "ARROW_DOWN:  display previous frame\n"
    "ARROW_RIGHT: display last frame\n"
    "ARROW_LEFT:  display first frame\n"
    "ESC:         exit the program (save " << FRAMES_SHARE << "OUT.ild)\n"
    "'o': increment selection start point\n"
    "'O': decrement selection start point\n"
    "'p': increment selection end point\n"
    "'P': decrement selection end point\n"
    "'l': increment palette index\n"
    "'L': decrement palette index\n"
    "'a': unblank all points in frame\n"
    "'b': blank selected points\n"
    "'B': unblank selected points\n"
    "'c': indicate selected start point index and color\n"
    "'C': recolor selected points with selected palette index\n"
    "'r': increment recolor effect and recolor frame\n"
    "'R': recolor all frames with 'recolor' effect\n"
    "'1': set 3D viewing angle to front\n"
    "'2': set 3D viewing angle to side\n"
    "'3': set 3D viewing angle to top\n"
    "'x': incremenmt 3D viewing angle X angle 1\n"
    "'X': decremenmt 3D viewing angle X angle 1\n"
    "'y': incremenmt 3D viewing angle Y angle 1\n"
    "'Y': decremenmt 3D viewing angle Y angle 1\n"
    "'z': incremenmt 3D viewing angle Z angle 1\n"
    "'Z': decremenmt 3D viewing angle Z angle 1\n"
    "'w': incremenmt 3D viewing angles X,Y,Z 1\n"
    "'W': decremenmt 3D viewing angles X,Y,Z 1\n"
    "'!': incremenmt 3D viewing angles X,Y,Z 1 and display next frame\n"
    "'@': decremenmt 3D viewing angles X,Y,Z 1 and display previous frame\n"
    "'t': transpose current frame to current viewing angle\n"
    "'T': transpose all frames to current viewing angle\n"
    "'/': set all Z to zero\n"
    "'?': set all Z to point order values\n"
    "'f': flip space left to right\n"
    "'F': flip space top to bottom\n"
    "'q': rotate space 90 clockwise\n"
    "'Q': rotate space 90 counterclockwise\n"
    "'s': scale all frames up 1%\n"
    "'S': scale all frames down 1%\n"
    "'u': reverse the order of frames\n"
    "'=': generate new frame(s) based on function\n"
    "'g': generate palette cycle animation\n"
    "     from here down, you figure it out!\n"
    "'G': wiggle.ild\n"
    "'h': out_2.ild\n"
    "'H': out_3.ild\n"
    "'j': out_4.ild\n"
    "'J': break.ild\n"
    "'k': out_7.ild\n"
    "'K': echos.ild\n"
    "'m': spin.ild\n"
    "'M': out_9.ild\n"
    "'n': half.ild\n"
    "'N': spin2.ild\n"
    "'9': span.ild\n"
    "'8': around.ild\n"
    "'7': around_.ild\n"
    "'6': stars.ild\n"
    "'0': tranz.ild\n"
    << endl;
    return;
}

//############################################################################
void show_some_hot_keys()
{
    cout << endl << endl
         << "*** Welcome to ild2fb! ***\n\n"
            "F1_KEY:      display whole list of hot keys\n"
            "F2_KEY:      open another ILDA file in " << FRAMES_SHARE << endl <<
            "F3_KEY:      toggle point select cursor visibility\n"
            "F4_KEY:      cycle global palette\n"
            "ARROW_UP:    display next frame, press and hold for animations\n"
            "ARROW_DOWN:  display previous frame\n"
            "ARROW_RIGHT: display last frame\n"
            "ARROW_LEFT:  display first frame\n"
            "'=':         generate new frame(s) based on function\n"
            "ESC:         exit the program (save OUT.ild)\n"
         << endl;
    return;
}

//############################################################################
int main(int argc, char** argv)
{
    ilda  ild; // this is it!
    int   error;

    if(argc > 1)
    {
        string new_file(argv[1]);
        if(    new_file.size() < 4
            || new_file.substr(new_file.size() - 4) != ".ild"
          )
            new_file += ".ild";
        error = ild.from_file(FRAMES_SHARE + new_file); // non zero is error
    }    
    else
        error = ild.from_file(FRAMES_SHARE + "in.ild"); // non zero is error

    u_int   key,
            recolor        =  6, // there are 7 different recolor effects
            section_num    =  0,
            point1         =  0,
            point2         =  0,
            global_palette =  0,
            palette_start  =  0,
            palette_end    = 62; // Pangolin

    u_char  palette_index  =  0;

    float ax    = 0.0,
          ay    = 0.0,
          az    = 0.0;

    bool  point_select_mode = true;

    ezfb_set_input_hot();
    //ofstream out("test.raw", ios::out | ios::binary);

    if(!ezfb_init_num(&fb, 0)) // as in /dev/fb0
    {
        cerr << "frame buffer failed to initialize" << endl;
        exit(1);
    }

    ezfb_set_bpp           (&fb, 8);
    ezfb_clear             (&fb);
    ezfb_set_cmap_pangolin (&fb);
    bmp_clone_ezfb_screen  (&fb,  &bmp, 0);

    show_some_hot_keys();

    if(error) // we still don't have any data
    {
        ild.append_section(ilda_section());
        cout << "\napplication started with NO DATA!" << endl;
    }

    ild[section_num].render_in_ezfb_bitmap_point_select(&bmp, point1, point2, ax, ay, az);
    ild[section_num].render_palette_in_ezfb_bitmap(&bmp, palette_index);
    bmp_to_ezfb(&fb, &bmp);

    while((key = ezfb_get_key()) != EZFB_ESCAPE_KEY) // from ezfb_tty.c
    { // ezfb_get_key() accepts a single key stroke with no return
        switch(key)
        {            
            //----------------------------------------------------------------
            case EZFB_F1_KEY:
            // show hot key list on terminal
                show_hot_keys();
                break;
            //----------------------------------------------------------------
            case EZFB_F2_KEY:
            // open another ILDA file
            {
                char buffer[81];
                cout << "\nenter new file name : ";
                ezfb_get_chars(buffer); // from ezfb_tty.c
                string new_file(buffer);
                if(    new_file.size() < 4
                    || new_file.substr(new_file.size() - 4) != ".ild"
                  )
                    new_file += ".ild";

                if(!ild.from_file(FRAMES_SHARE + new_file)) // non zero is error
                {
                    section_num = 0;
                    point1      = 0;
                    point2      = 0;
                }
                cout << ild.file_name << " now open" << endl;
                break;
            }
            //----------------------------------------------------------------
            case EZFB_F3_KEY:
            // toggle point select cursor visibility
                point_select_mode = !point_select_mode;    
                break;
            //----------------------------------------------------------------
            case EZFB_F4_KEY:
            // cycle global palette
                global_palette++;
                if(global_palette >= 5)
                    global_palette = 0;
                switch(global_palette)
                {
                    case  0: ezfb_set_cmap_pangolin(&fb);
                             palette_start  =   0,
                             palette_end    =  62;
                             cout << "Pangolin palette" << endl;
                             break;
                    case  1: ezfb_set_cmap_laser_illusions(&fb);
                             palette_start  =   1,
                             palette_end    = 254;                             
                             cout << "Laser Illusions palette" << endl;
                             break;
                    case  2: ezfb_set_cmap_x29(&fb);
                             palette_start  =   2,
                             palette_end    =  19;                             
                             cout << "X29 palette" << endl;
                             break;
                    case  3: ezfb_set_cmap_ilda_standard(&fb);
                             palette_start  =   1,
                             palette_end    = 254;                             
                             cout << "ILDA Standard palette" << endl;
                             break;
                    case  4: ezfb_set_cmap_hues(&fb);
                             palette_start  =   2,
                             palette_end    = 253;                             
                             cout << "EZFB Hues palette" << endl;
                             break;
                    default: cout << "impossible!" << endl;
                             break;

                }
                bmp_palette_from_ezfb(&fb, &bmp);
                break;
            //----------------------------------------------------------------
            case EZFB_ARROW_UP:
            // display next frame, press and hold for animations
                section_num++;
                if(section_num >= ild.num_sections)
                    section_num = 0;
                break;
            //----------------------------------------------------------------
            case EZFB_ARROW_DOWN:
            // display previous frame
                section_num--;
                if((int)section_num < 0)
                    section_num = ild.num_sections - 1;
                break;
            //----------------------------------------------------------------
            case EZFB_ARROW_RIGHT:
            // display last frame
                section_num = ild.num_sections - 1;
                break;
            //----------------------------------------------------------------
            case EZFB_ARROW_LEFT:
            // display first frame
                section_num = 0;
                break;
            //----------------------------------------------------------------







            //----------------------------------------------------------------
            // increment selection start point
            case 'o': point1++;
                         break;
            // decrement selection start point
            case 'O': point1--;
                         break;
            //----------------------------------------------------------------
            // increment selection end point
            case 'p': point2++;
                         break;
            // decrement selection end point
            case 'P': point2--;
                         break;
            //----------------------------------------------------------------
            // increment palette index
            case 'l': palette_index++;
                         break;
            // decrement palette index
            case 'L': palette_index--;
                         break;
            //----------------------------------------------------------------




            
            
                        
            //----------------------------------------------------------------
            // unblank all points in frame
            case 'a':
            {
                point1 = 0;
                for(u_int frame = 0; frame < ild.number_of_sections(); frame++)
                {
                    point2 = ild[frame].quantity - 1;
                    ild[frame].unblank_points(point1, point2);
                }
                break;
            }            
            //----------------------------------------------------------------
            // blank selected points
            case 'b':
                ild[section_num].blank_points(point1, point2);
                break;
            // unblank selected points
            case 'B':
                ild[section_num].unblank_points(point1, point2);                
                break;
            //----------------------------------------------------------------
            // indicate selected start point index and color
            case 'c': cout << "\npoint "
                                     << point1
                                     << " is palette index : "
                                     << (int)(ild[section_num].get_point_color(point1));
                break;
            //----------------------------------------------------------------
            // recolor selected points with selected palette index
            case 'C':
                ild[section_num].set_points_color(point1, point2, palette_index);                                
                break;
            //----------------------------------------------------------------
            // increment recolor effect and recolor frame
            case 'r':
                recolor++;
                if(recolor >= 7)
                    recolor = 0;
                ild[section_num].rainbow_recolor(recolor, palette_start, palette_end, 2);
                switch(recolor)
                {
                    case  0: cout << "span through x"       << endl; break;
                    case  1: cout << "span through y"       << endl; break;
                    case  2: cout << "span through z"       << endl; break;
                    case  3: cout << "span through points"  << endl; break;
                    case  4: cout << "index through points" << endl; break;
                    case  5: cout << "index through shapes" << endl; break;
                    case  6: cout << "span through shapes"  << endl; break;
                    default: cout << "impossible!"          << endl; break;
                }
                break;
            //----------------------------------------------------------------
            // recolor all frames with 'recolor' effect
            case 'R':
            {
                for(u_int i = 0; i < ild.number_of_sections(); i++)
                    ild[i].rainbow_recolor(recolor, palette_start, palette_end, 2);
                cout << "\nDONE: recolor all frames" << endl;
                break;
            }
            //----------------------------------------------------------------







            //----------------------------------------------------------------
            // set 3D viewing angle to front
            case '1':
                ax = 0;
                ay = 0;
                az = 0;
                break;
            // set 3D viewing angle to side
            case '2':
                ax = half_pi;
                ay = 0;
                az = 0;
                break;
            // set 3D viewing angle to top
            case '3':
                ax = 0;
                ay = half_pi;
                az = 0;
                break;
            //----------------------------------------------------------------
            // incremenmt 3D viewing angle X angle 1
            case 'x': ax += one_degree;
                       break;
            // decremenmt 3D viewing angle X angle 1
            case 'X': ax -= one_degree;
                       break;
            //----------------------------------------------------------------
            // incremenmt 3D viewing angle Y angle 1
            case 'y': ay += one_degree;
                       break;
            // decremenmt 3D viewing angle Y angle 1
            case 'Y': ay -= one_degree;
                       break;
            //----------------------------------------------------------------
            // incremenmt 3D viewing angle Z angle 1
            case 'z': az += one_degree;
                       break;
            // decremenmt 3D viewing angle Z angle 1
            case 'Z': az -= one_degree;
                       break;
            //----------------------------------------------------------------
            // incremenmt 3D viewing angles X,Y,Z angle 1
            case 'w':
                ax += one_degree;
                ay += one_degree;
                az += one_degree;
                break;
            // decremenmt 3D viewing angles X,Y,Z angle 1
            case 'W':
                ax -= one_degree;
                ay -= one_degree;
                az -= one_degree;
                break;
            //----------------------------------------------------------------
            // transpose current frame to current viewing angle
            case 't':
                ild[section_num].rotate_all_around_origin(ax, ay, az);
                ax = 0;
                ay = 0;
                az = 0;
                break;
            // transpose all frames to current viewing angle
            case 'T':
            {
                for(u_int i = 0; i < ild.num_sections; i++)
                {
                    ild[i].rotate_all_around_origin(ax, ay, az);
                }
                ax = 0;
                ay = 0;
                az = 0;
                cout << "\nDONE: 3D transpose all frames" << endl;
                break;
            }
            //----------------------------------------------------------------
            // incremenmt 3D viewing angles X,Y,Z angle 1 and display next frame
            case '!':
                ax += one_degree;
                ay += one_degree;
                az += one_degree;
                section_num++;
                if(section_num >= ild.num_sections)
                    section_num = 0;
                break;
            //----------------------------------------------------------------
            // decremenmt 3D viewing angles X,Y,Z angle 1 and display previous frame
            case '@':
                ax -= one_degree;
                ay -= one_degree;
                az -= one_degree;
                section_num--;
                if(section_num < 0)
                    section_num = ild.num_sections - 1;
                break;
            //----------------------------------------------------------------









            //----------------------------------------------------------------
            // set all Z to zero
            case '/': ild[section_num].flatten_z();
                      break;
            // set all Z to point order values
            case '?': ild[section_num].z_order_points(17000);
                      break;
            //----------------------------------------------------------------
            // flip space left to right
            case 'f': ild[section_num].flip(0);
                      break;
            // flip space top to bottom
            case 'F': ild[section_num].flip(1);
                      break;
            //----------------------------------------------------------------
            // rotate space 90 clockwise
            case 'q': ild[section_num].quarter_turn(0, 1);
                      break;
            // rotate space 90 counterclockwise
            case 'Q': ild[section_num].quarter_turn(0, 3);
                      break;
            //----------------------------------------------------------------







            //----------------------------------------------------------------
            // scale all frames up 1%
            case 's':
            {
                for(u_int i = 0; i < ild.number_of_sections(); i++)
                    ild[i].scale(1.01, 1.01, 1.01);
                break;
            }
            //----------------------------------------------------------------
            // scale all frames down 1% : DISTROYS RESOLUTION!
            case 'S':
            {
                for(u_int i = 0; i < ild.number_of_sections(); i++)
                    ild[i].scale(0.99, 0.99, 0.99);
                break;
            }
            //----------------------------------------------------------------






                
            //----------------------------------------------------------------
            // reverse the order of all frames
            case 'u': ild.reverse();
                      break;
            //----------------------------------------------------------------






            
            //----------------------------------------------------------------
            case '=':
            // generate new frame(s) based on function ilda_3D_point(*F)(int total_points, int frame_num)
            {
                //ild.append_section(ilda_section(A, 80, 0));

                //for(int frame = 0; frame < 60; frame++)
                //    ild.append_section(ilda_section(B, 20, frame));

                ild.append_section(ilda_section(F, 700, 0));

                section_num = ild.number_of_sections() - 1;
                break;
            }
            //----------------------------------------------------------------            

            
            
            
            
                        
            
            //----------------------------------------------------------------
            // generate palette cycle animation
            case 'g':
            {
                ilda out;
                out.append_section(ild[section_num]);
                for(u_int i = 0; i < (palette_end - palette_start); i++)
                {
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "rainbow.ild");
                cout << "\nSAVED: rainbow.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            // from here down, you figure it out!
            // All of these transforms are pretty much the way I left them.
            // when I was done with the 2004 First Night Akron show.
            //----------------------------------------------------------------
            case 'G':
            {
                ilda out;
                out.append_section(ild[section_num]);
                for(u_int i = 0; i < 99; i++)
                {
                    ild[section_num].wiggle_shapes(3, 6000);
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "wiggle.ild");
                cout << "\nSAVED: wiggle.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'h':
            {
                ilda out;
                out.append_section(ild[section_num]);
                for(u_int i = 0; i < 199; i++)
                {
                    ild[section_num].roto_wiggle_shapes(3, pi/50.0);
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "out_2.ild");
                cout << "\nSAVED: out_2.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'H':
            {
                ilda out;
                out.append_section(ild[section_num]);
                for(u_int i = 0; i < 399; i++)
                {
                    ild[section_num].roto_wiggle_shapes(3, pi/30.0);
                    ild[section_num].wiggle_shapes(3, 900);
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "out_3.ild");
                cout << "\nSAVED: out_3.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'j':
            {
                ilda out;
                out.append_section(ild[section_num]);
                for(u_int i = 0; i < 299; i++)
                {
                    ild[section_num].falling_shapes(0, 1700);
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "out_4.ild");
                cout << "\nSAVED: out_4.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'J':
            {
                u_int frame, shape;
                int   out_of_bounds;
                ilda out;
                short *dx = new short[ild[section_num].number_of_shapes()];
                short *dy = new short[ild[section_num].number_of_shapes()];
                short *dz = new short[ild[section_num].number_of_shapes()];
                for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                {
                    dx[shape] = short((ezfb_random_01() - 0.5) * 2000);
                    dy[shape] = short((ezfb_random_01() - 0.5) * 2000);
                    dz[shape] = short((ezfb_random_01() - 0.5) * 2000);
                }
                out.append_section(ild[section_num]);
                for(frame = 0; frame < 299; frame++)
                {
                    for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                        while((out_of_bounds = ild[section_num].move_shape(shape, dx[shape], dy[shape], dz[shape])))
                        {
                            switch(out_of_bounds)
                            {
                                case 1: dx[shape] = -dx[shape]; break;
                                case 2: dy[shape] = -dy[shape]; break;
                                case 3: dz[shape] = -dz[shape]; break;
                            }
                        }
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                }
                out.reverse();
                out.save_as(FRAMES_SHARE + "break.ild");
                cout << "\nSAVED: break.ild" << endl;
                delete dx;
                delete dy;
                delete dz;
                break;
            }
            //----------------------------------------------------------------
            case 'k':
            {
                u_int frame, shape;
                int   out_of_bounds;
                ilda out;
                float *ax = new float[ild[section_num].number_of_shapes()];
                float *ay = new float[ild[section_num].number_of_shapes()];
                float *az = new float[ild[section_num].number_of_shapes()];
                for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                {
                    ax[shape] = (ezfb_random_01() - 0.5) * pi/20;
                    ay[shape] = (ezfb_random_01() - 0.5) * pi/20;
                    az[shape] = (ezfb_random_01() - 0.5) * pi/20;
                }

                out.append_section(ild[section_num]);
                for(frame = 0; frame < 59; frame++)
                {
                    for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                        while((out_of_bounds = ild[section_num].rotate_shape(shape, ax[shape], ay[shape], az[shape])))
                        {
                            switch(out_of_bounds)
                            {
                                case 1: ax[shape] = -ax[shape]; break;
                                case 2: ay[shape] = -ay[shape]; break;
                                case 3: az[shape] = -az[shape]; break;
                            }
                        }
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                }
                out.save_as(FRAMES_SHARE + "out_7.ild");
                cout << "\nSAVED: out_7.ild" << endl;
                delete ax;
                delete ay;
                delete az;
                break;
            }
            //----------------------------------------------------------------
            case 'K':
            {
                u_int   echo,
                        delay =  1,
                        num_echos = 11;
                ilda out;
                for(u_int i = 0; i < (ild.number_of_sections() + delay * num_echos); i++)
                {
                    if(i < ild.number_of_sections())
                        out.append_section(ild[i]);
                    else
                        out.append_section();
                    for(echo = 1; echo <= num_echos; echo++)
                        if(i >= (delay * echo) && (i - delay * echo < ild.number_of_sections()))
                            out[i] += ild[i - delay * echo];
                }
                out.save_as(FRAMES_SHARE + "echos.ild");
                cout << "\nSAVED : echos.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'm':
            {
                u_int frame,
                      shape;
                int   out_of_bounds;

                float *rx = new float[ild[section_num].number_of_shapes()];
                float *ry = new float[ild[section_num].number_of_shapes()];
                float *rz = new float[ild[section_num].number_of_shapes()];
                short *dx = new short[ild[section_num].number_of_shapes()];
                short *dy = new short[ild[section_num].number_of_shapes()];
                short *dz = new short[ild[section_num].number_of_shapes()];

                ilda out;

                for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                {
                    dx[shape] = short(ezfb_random_neg_to_pos_1() * 900);
                    dy[shape] = short(ezfb_random_neg_to_pos_1() * 900);
                    dz[shape] = short(ezfb_random_neg_to_pos_1() * 300);
                    rx[shape] = ezfb_random_neg_to_pos_1() * pi/8.0;
                    ry[shape] = ezfb_random_neg_to_pos_1() * pi/8.0;
                    rz[shape] = ezfb_random_neg_to_pos_1() * pi/8.0;
                }

                out.append_section(ild[section_num]);
                for(frame = 0; frame < 599; frame++)
                {
                    for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                    {
                        ild[section_num].rotate_shape(shape, rx[shape], ry[shape], rz[shape]);
                        while((out_of_bounds = ild[section_num].move_shape(shape, dx[shape], dy[shape], dz[shape])))
                        {
                            switch(out_of_bounds)
                            {
                                case 1: dx[shape] = -dx[shape]; break;
                                case 2: dy[shape] = -dy[shape]; break;
                                case 3: dz[shape] = -dz[shape]; break;
                            }
                        }
                    }
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                }
                out.save_as(FRAMES_SHARE + "spin.ild");
                cout << "\nSAVED : spin.ild" << endl;

                delete dx;
                delete dy;
                delete dz;
                delete rx;
                delete ry;
                delete rz;
                break;
            }
            //----------------------------------------------------------------
            case 'M':
            {
                u_int   i_frame = 0,
                        o_frame;
                short   dx =  22000,
                        dy = -16000;
                ilda    out;
                for(o_frame = 0; o_frame < 299; o_frame++)
                {
                    out.append_section(ild[i_frame]);
                    out[o_frame].scale(0.25, 0.25, 0.25);
                    if(out[o_frame].move(dx, dy, 0))
                        break;
                    dx -= 240;
                    i_frame++;
                    if(i_frame >= ild.number_of_sections())
                        i_frame = 0;
                }
                out.save_as(FRAMES_SHARE + "out_9.ild");
                cout << "\nSAVED: out_9.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'n':
            {
                ilda  out;
                for(u_int frame = 0; frame < ild.number_of_sections(); frame++)
                {
                    if(frame % 2 == 0)
                        out.append_section(ild[frame]);
                }
                out.save_as(FRAMES_SHARE + "half.ild");
                cout << "\nSAVED: half.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case 'N':
            {
                int  o_frame;
                ilda out;
                ild[section_num].rotate_all_around_origin(0, pi/2, 0);
                for(o_frame = 0; o_frame < 7; o_frame++)
                {
                    out.append_section(ild[section_num]);
                    ild[section_num].rotate_all_around_origin(-pi/120, 0, 0);
                }
                for(o_frame = 0; o_frame < 180; o_frame++)
                {
                    out.append_section(ild[section_num]);
                    ild[section_num].rotate_all_around_origin(0, pi/120, 0);
                }
                for(o_frame = 0; o_frame < 7; o_frame++)
                {
                    out.append_section(ild[section_num]);
                    ild[section_num].rotate_all_around_origin(0, 0, pi/120);
                }
                out.save_as(FRAMES_SHARE + "spin2.ild");
                cout << "\nSAVED: spin2.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case '9':
            {
                u_int frame,
                      shape;
                int   out_of_bounds,
                      dx = 1000,
                      dy = 1000;
                ilda out;
                out.append_section(ild[section_num]);
                for(frame = 0; frame < 59; frame++)
                {
                    for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                    {
                        ild[section_num].rotate_shape(shape, 0, 0, pi/30);
                        while((out_of_bounds = ild[section_num].move_shape(shape, dx, dy, 0)))
                        {
                            switch(out_of_bounds)
                            {
                                case 1: dx = -dx; break;
                                case 2: dy = -dy; break;
                            }
                        }
                    }
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                }
                out.save_as(FRAMES_SHARE + "span.ild");
                cout << "\nSAVED: span.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case '8':
            {
                u_int frame, shape;
                ilda out;
                out.append_section(ild[section_num]);
                for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                {
                    for(frame = 0; frame < 5; frame++)
                    {
                        ild[section_num].rotate_shape(shape, 0, 0, pi/10);
                        out.append_section(ild[section_num]);
                    }
                }
                for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                {
                    for(frame = 0; frame < 10; frame++)
                    {
                        ild[section_num].rotate_shape(shape, 0, 0, pi/10);
                        out.append_section(ild[section_num]);
                    }
                }
                for(frame = 0; frame < 10; frame++)
                {
                    for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                        ild[section_num].rotate_shape(shape, 0, 0, pi/10);
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "around.ild");
                cout << "\nSAVED: around.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case '7':
            {
                u_int  spoke;
                ilda   out;
                
                for(u_int frame = 0; frame < ild.number_of_sections(); frame++)
                {
                    out.append_section(ild[frame]);
                    for(spoke = 1; spoke < 11; spoke++)
                    {
                        out[frame].rotate_all_around_origin(0, 0, pi/5.5);
                        out[frame] += ild[frame];
                    }
                    out[frame].palette_cycle(palette_start, palette_end, 1);
                }
                out.save_as(FRAMES_SHARE + "around_.ild");
                cout << "\nSAVED: around_.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            // Star field explosion only works on a set of individual points
            //----------------------------------------------------------------
            case '6':
            {
                ilda    out;
                u_int   frame, shape;
                short   cx, cy, cz, dx, dy, dz;
                for(frame = 0; frame < 699; frame++)
                {
                    for(shape = 0; shape < ild[section_num].number_of_shapes(); shape++)
                    {
                        if(ild[section_num].scale_shape_around_origin(shape, 1.1, 1.1, 1.1)) // out of bounds
                        {
                            ild[section_num].find_shape_center(shape, cx, cy, cz);
                            while(!(dx = rand() % 5000 - 2500)); //  -2500 ~ +2500, not zero
                            while(!(dy = rand() % 5000 - 2500));
                            while(!(dz = rand() % 5000 - 2500));
                            ild[section_num].move_shape(shape, -cx + dx, -cy + dy, -cz + dz); // somewhere near the origin
                        }
                    }
                    out.append_section(ild[section_num]);
                }
                out.save_as(FRAMES_SHARE + "stars.ild");
                cout << "\nSAVED : stars.ild" << endl;
                break;
            }
            //----------------------------------------------------------------
            case '5': // adding static frame to a motion set
            {
                ilda blimp("blimp.ild");
                for(u_int i = 0; i < ild.number_of_sections(); i++)
                {
                    ild[i].scale(0.3, 0.3, 0.3);
                    //ild[i].move(0,-2400, 0);
                    ild[i] += blimp[1];
                }
                break;
            }
            //----------------------------------------------------------------
            case '0':
            {
                u_int i;
                ilda out;
                float amplitude = 0;
                float phase = pi/2;
                out.append_section(ild[section_num]);
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].concentric_sine_ripple(amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude += (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].concentric_sine_ripple(amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].concentric_sine_ripple(amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude -= (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(0, amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude += (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(0, amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(0, amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude -= (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(1, amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude += (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(1, amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(1, amplitude, 2, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude -= (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(2, amplitude, 4, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude += (.66 / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(2, amplitude, 4, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                }
                for(i = 0; i < 60; i++)
                {
                    ild[section_num].parallel_sine_ripple(2, amplitude, 4, phase);
                    out.append_section(ild[section_num]);
                    ild[section_num].palette_cycle(palette_start, palette_end, 1);
                    phase += (two_pi / 61.0);
                    amplitude -= (.66 / 61.0);
                }
                out.save_as(FRAMES_SHARE + "tranz.ild");
                cout << "\nSAVED: tranz.ild" << endl;
                break;
            }
        //--------------------------------------------------------------------
        } // end switch(key)


        if(point_select_mode)
            ild[section_num].render_in_ezfb_bitmap_point_select(&bmp, point1, point2, ax, ay, az);
        else
            ild[section_num].render_in_ezfb_bitmap(&bmp, ax, ay, az);

        ild[section_num].render_palette_in_ezfb_bitmap(&bmp, palette_index);
        bmp_to_ezfb(&fb, &bmp);
    } // while((key = ezfb_get_key()) != EZFB_ESCAPE_KEY)
    cout << "\napplication ezfb_terminated!" << endl;
    ild.save_as  (FRAMES_SHARE + "OUT.ild"); // save possibly altered ILDA file
    ezfb_release (&fb);
    bmp_free     (&bmp);
    //out.close();
    return 0;
}

//############################################################################
//////////////////////////////////////////////////////////////////////////////
//############################################################################
