//############################################################################
// tsc ~ Touch Screen Calibration Application
//
// James Lehman
// Akron, Ohio
//
// james@akrobiz.com
//
// This application requires ezfb frame buffer API
// also written by James Lehman.
//
// NOTE: Requires "/dev/tsraw" !!!
// the ability to read raw data from the touch screen device.
//
// Adapted from xtscal ~ touchscreen calibration application
// Adapted from the h3600 X calibration Application
// Adapted from the Itsy calibration app by C Flynn
//
// Use consistent with the GNU GPL is permitted,
// provided that this copyright notice is
// preserved in its entirety in all copies and derived works.
//
// JAMES LEHMAN / EXTRA STIMULUS INC. MAKES NO WARRANTIES,
// EXPRESSED OR IMPLIED, AS TO THE USEFULNESS OR CORRECTNESS
// OF THIS CODE OR ITSFITNESS FOR ANY PARTICULAR PURPOSE.
//
//############################################################################

#include "ezfb.h"

//############################################################################
#define WITH_LOGO                   1  // make a logo smaller than your screen.
#define LOGO_LOCATION               "/usr/local/share/tsclogo.bmp" // put it here.

#define TOUCH_SCREEN_CAL_DEBUG      0

#define UCB1200_TS_SET_RAW_MAX_X    3  // These should all be in
#define UCB1200_TS_SET_RAW_MIN_X    10 // the touch screen driver,
#define UCB1200_TS_SET_RAW_MAX_Y    4  // but they are not!
#define UCB1200_TS_SET_RAW_MIN_Y    11
#define UCB1200_TS_SET_XY_SWAP      12
#define UCB1200_TS_SET_CAL_DEFAULTS 18

#define RAW_DEV_NODE                "/dev/tsraw"
#define DEV_NODE                    "/dev/ts"
#define TSC_CONFIG_FILE             "/etc/tsc.conf"

//############################################################################
struct ezfb fb = {0}; // global frame buffer object

//############################################################################
typedef struct
{
    ushort x;
    ushort y;
}   TS_POINT;

//############################################################################
typedef struct
{
    ushort pressure ;
    ushort x        ;
    ushort y        ;
    ushort millisecs;
}   TS_EVENT        ;

//############################################################################
int    ts_dev_fd      = 0 ;
int    raw_max_x          ;
int    raw_max_y          ;
int    raw_min_x          ;
int    raw_min_y          ;
int    xy_swap        = 0 ;
float  xscale             ;
float  yscale             ;

//############################################################################
void   do_default_operations (                                      );
void   commit_config_to_file (char* conf_file                       );
void   load_config_from_file (char* conf_file                       );
void   fb_startup            (                                      );
void   ts_open               (char* dev_name                        );
void   ts_close              (                                      );
void   show_usage            (char* prog_name                       );
void   do_calibration        (                                      );
void   do_test               (                                      );
int    prompt_for_test       (                                      );
void   drawPlus              (int xcoord, int ycoord, int cmap_index);
void   drawX                 (int xcoord, int ycoord, int cmap_index);
void   drawBeams             (int xcoord, int ycoord, int cmap_index);
void   drawTEST              (int xcoord, int ycoord, int cmap_index);
void   drawEXIT              (int xcoord, int ycoord, int cmap_index);
void   set_driver_parms      (                                      );

//############################################################################
int main(int argc, char **argv)
{
    if(argc > 1)  // some args entered at command line
    {
        if(!strcmp(argv[1], "--help"))
            show_usage(argv[0]);
        //-------------------------------------------------
        else if(!strcmp(argv[1], "--test"))
        {
            ts_open(DEV_NODE);
            fb_startup();
            do_test();
            ts_close();
            ezfb_release(&fb);
        }
        //-------------------------------------------------
        else if(!strcmp(argv[1], "--cal"))
        {   if(argc < 6)
            {
                fprintf(stderr, "\nERROR: too few arguments for --cal");
                show_usage(argv[0]);
                exit(1);
            }
            sscanf(argv[2], "%d", &raw_max_x);
            sscanf(argv[3], "%d", &raw_max_y);
            sscanf(argv[4], "%d", &raw_min_x);
            sscanf(argv[5], "%d", &raw_min_y);
            sscanf(argv[6], "%d", &xy_swap  );
            ts_open(RAW_DEV_NODE);
            set_driver_parms();
            ts_close();
        }
        //-------------------------------------------------
        else if(!strcmp(argv[1], "--reset"))
        {
            ts_open(RAW_DEV_NODE);
            ioctl(ts_dev_fd, UCB1200_TS_SET_CAL_DEFAULTS, NULL);
            ts_close();
        }
        //-------------------------------------------------
        else if(!strcmp(argv[1], "--commit"))
        {
            do_default_operations();

            if(argc > 2)
                commit_config_to_file(argv[2]);
            else
                commit_config_to_file(TSC_CONFIG_FILE);
        }
        //-------------------------------------------------
        else if(!strcmp(argv[1], "--load"))
        {
            if(argc > 2)
                load_config_from_file(argv[2]);
            else
                load_config_from_file(TSC_CONFIG_FILE);
        }
    }
    //-------------------------------------------------
    else /* no args entered at command line */
        do_default_operations();

    return 0;
}

//############################################################################
void do_default_operations()
{
    ts_open(RAW_DEV_NODE);
    fb_startup      ();
    do_calibration  ();
    set_driver_parms();
    ezfb_clear      (&fb);
    if(prompt_for_test())
    {
        ts_close();
        ts_open(DEV_NODE);
        do_test();
    }
    ezfb_clear(&fb);
    ts_close();
    ezfb_release(&fb);
}

//############################################################################
void commit_config_to_file(char* conf_file_name)
{
    FILE* conf_file;

    conf_file = fopen(conf_file_name, "w");
    if(!conf_file)
    {
        perror("\ntsc ERROR: opening output file failed");
        exit(1);
    }
    fprintf(conf_file, "# touch screen calibration config\n");
    fprintf(conf_file, "# auto-generated by tsc\n\n");
    fprintf(conf_file, "RAW_MAX_X: %d\n", raw_max_x);
    fprintf(conf_file, "RAW_MAX_Y: %d\n", raw_max_y);
    fprintf(conf_file, "RAW_MIN_X: %d\n", raw_min_x);
    fprintf(conf_file, "RAW_MIN_Y: %d\n", raw_min_y);
    fprintf(conf_file, "XY_SWAP: %d\n"  , xy_swap  );
    printf("\nRAW_MAX_X: %d\nRAW_MAX_Y: %d\nRAW_MIN_X: %d\nRAW_MIN_Y: %d\nXY_SWAP: %d\nts config saved in file: %s\n\n", raw_max_x, raw_max_y, raw_min_x, raw_min_y, xy_swap, conf_file_name);
    fclose(conf_file);
    return;
}

//############################################################################
void load_config_from_file(char* conf_file_name)
{
    FILE* conf_file;
    int   all_is_well = 1;
    char  stuff_from_file[100];

    conf_file = fopen(conf_file_name, "r");
    if(!conf_file)
    {
        perror("\ntsc ERROR: opening input file failed");
        exit(1);
    }

    do
    {
        fgets(stuff_from_file, 99, conf_file);
    } while(stuff_from_file[0] == '#');

    all_is_well &= fscanf(conf_file, " RAW_MAX_X: %d ", &raw_max_x);
    all_is_well &= fscanf(conf_file, " RAW_MAX_Y: %d ", &raw_max_y);
    all_is_well &= fscanf(conf_file, " RAW_MIN_X: %d ", &raw_min_x);
    all_is_well &= fscanf(conf_file, " RAW_MIN_Y: %d ", &raw_min_y);
    all_is_well &= fscanf(conf_file, " XY_SWAP: %d "  , &xy_swap  );

    if(all_is_well)
    {
        ts_open(RAW_DEV_NODE);
        set_driver_parms();
        ts_close();
        printf("\nRAW_MAX_X: %d\nRAW_MAX_Y: %d\nRAW_MIN_X: %d\nRAW_MIN_Y: %d\nXY_SWAP: %d\nts config set from file: %s sucessful.\n\n", raw_max_x, raw_max_y, raw_min_x, raw_min_y, xy_swap, conf_file_name);
    }
    else
    {
        printf("ts config FAILED from file: %s", conf_file_name);
    }

    fclose(conf_file);
    return;
}

//############################################################################
void fb_startup()
{
    if(0 == ezfb_init(&fb))
        exit(1);
    ezfb_clear(&fb);

#ifdef WITH_LOGO
    bmp_file_to_ezfb_center(&fb, LOGO_LOCATION);
#endif

    return;
}

//############################################################################
void ts_open(char* dev_name)
{
    if((ts_dev_fd = open(dev_name, O_RDONLY)) < 0)
    {
        perror("ERROR: unable to open touch screen device.");
        exit(1);
    }
    return;
}

//############################################################################
void ts_close()
{
    if(ts_dev_fd)
        close(ts_dev_fd);
    return;
}

//############################################################################
void show_usage(char* prog_name)
{
    printf("\nUSAGE: %s <optional arguments>\n", prog_name                                    );
    printf("   <no args>         set new session calibration\n"                               );
    printf("  optional args:\n"                                                               );
    printf("    --help           print help and exit [must be root to do anything else!]\n"   );
    printf("    --cal <max_x> <max_y> <min_x> <min_y> <xy_swap>\n"                            );
    printf("    --reset          reset driver to default (fairly close) values\n"             );
    printf("    --test           show where screen is pressed (5 hits to exit)\n"             );
    printf("    --commit [file]  write configuration to [file] /etc/tsc.conf by default\n"    );
    printf("    --load [file]    read configuration from [file] /etc/tsc.conf by default\n\n" );
    exit(0);
}

//############################################################################
void do_calibration()
{
    int              k, xsum, ysum, cnt, rv, x_diff, y_diff, offset;
    TS_EVENT         ts_ev;
    TS_POINT         display_point     [4];
    int              averaged_raw_in_a [4];
    int              averaged_raw_in_b [4];
    int              *x;
    int              *y;

    offset = fb.Var.xres / 20;

    display_point[0].x = offset;
    display_point[0].y = offset;

    display_point[1].x = fb.Var.xres  - offset - 1;
    display_point[1].y = offset;

    display_point[2].x = offset;
    display_point[2].y = fb.Var.yres - offset - 1;

    display_point[3].x = fb.Var.xres  - offset - 1;
    display_point[3].y = fb.Var.yres - offset - 1;

    x_diff             = display_point[1].x - display_point[0].x;
    y_diff             = display_point[2].y - display_point[0].y;

    for(k = 0; k < 4 ; k++)
    {
#if TOUCH_SCREEN_CAL_DEBUG
        printf("\nTouch screen at x=%d y=%d\n", display_point[k].x, display_point[k].y);
#endif
        drawPlus(display_point[k].x, display_point[k].y, 255); // white
        cnt = xsum = ysum = 0;
        while(1)
        {
            rv = read(ts_dev_fd, &ts_ev, sizeof(TS_EVENT));
            if(ts_ev.pressure == 0  &&  cnt > 5)
                break;
            xsum += ts_ev.y;
            ysum += ts_ev.x;
#if TOUCH_SCREEN_CAL_DEBUG
            printf("raw point:  (%d,%d)\n", ts_ev.x, ts_ev.y);
#endif
            cnt++;
            drawPlus (display_point[k].x, display_point[k].y, 0); // black
            drawX    (display_point[k].x, display_point[k].y, 255); // white
        }
        drawX(display_point[k].x, display_point[k].y, 253); // red
        averaged_raw_in_a[k] = xsum / cnt;
        averaged_raw_in_b[k] = ysum / cnt;
#if TOUCH_SCREEN_CAL_DEBUG
        printf(" k=%d  AveX=%3d  AveY=%3d  (cnt=%3d)\n", k, averaged_raw_in_a[k], averaged_raw_in_b[k], cnt);
#endif
    }

    xy_swap   = (abs(averaged_raw_in_a[1] - averaged_raw_in_a[0]) > abs(averaged_raw_in_b[1] - averaged_raw_in_b[0])) ? (0) : (1);

    if(xy_swap)
    {
        x = averaged_raw_in_b;
        y = averaged_raw_in_a;
    }
    else
    {
        x = averaged_raw_in_a;
        y = averaged_raw_in_b;
    }

    xscale = (   ((float)(x_diff) / (float)(x[1] - x[0]))
               + ((float)(x_diff) / (float)(x[1] - x[2]))
               + ((float)(x_diff) / (float)(x[3] - x[0]))
               + ((float)(x_diff) / (float)(x[3] - x[2]))  ) / 4.0;

    yscale = (   ((float)(y_diff) / (float)(y[2] - y[0]))
               + ((float)(y_diff) / (float)(y[2] - y[1]))
               + ((float)(y_diff) / (float)(y[3] - y[0]))
               + ((float)(y_diff) / (float)(y[3] - y[1]))  ) / 4.0;
#if TOUCH_SCREEN_CAL_DEBUG
    printf("xscale                   : %f\n", xscale);
    printf("yscale                   : %f\n", yscale);
#endif

    raw_max_x = (int)((   ((float)(x[1]) + ((float)(offset) / xscale))
                        + ((float)(x[3]) + ((float)(offset) / xscale)) ) / 2.0);

    raw_max_y = (int)((   ((float)(y[2]) + ((float)(offset) / yscale))
                        + ((float)(y[3]) + ((float)(offset) / yscale)) ) / 2.0);


    raw_min_x = (int)((   ((float)(x[0]) - ((float)(offset) / xscale))
                        + ((float)(x[2]) - ((float)(offset) / xscale)) ) / 2.0);

    raw_min_y = (int)((   ((float)(y[0]) - ((float)(offset) / yscale))
                        + ((float)(y[1]) - ((float)(offset) / yscale)) ) / 2.0);

    return;
}
//############################################################################
void do_test()
{
    int              k;
    TS_EVENT         ts_ev;
    ushort           old_x = 0xffff, old_y = 0xffff;

    for(k = 0; k < 5 ; k++)
    {
        while(1)
        {
            read(ts_dev_fd, &ts_ev, sizeof(TS_EVENT));
            drawBeams(old_x, old_y, 0); // black
            old_x = ts_ev.x;
            old_y = ts_ev.y;
            drawBeams(ts_ev.x, ts_ev.y, 255); // white
            if (ts_ev.pressure == 0)
                break;
        }
    }
    return;
}

//############################################################################
int prompt_for_test()
{
    TS_EVENT         ts_ev;
    int              i, j, x, y, x1, x2, y1, y2;

    x1 = 2 * (fb.Var.xres / 5) + (fb.Var.xres / 20) - 5;
    x2 = 2 * (fb.Var.xres / 5) + (fb.Var.xres / 20) + (21 * (fb.Var.xres / 160)) + 5;
    y1 = 3 * (fb.Var.xres / 9) + (fb.Var.xres / 80) - 5;
    y2 = 3 * (fb.Var.xres / 9) + (fb.Var.xres / 80) + ( 6 * (fb.Var.xres / 160)) + 5;

    ezfb_put_line(&fb, x1, y1, x1, y2, 255); // white
    ezfb_put_line(&fb, x2, y1, x2, y2, 255);
    ezfb_put_line(&fb, x1, y1, x2, y1, 255);
    ezfb_put_line(&fb, x1, y2, x2, y2, 255);

    for(i = 0; i < 5; i++)
        for(j = 0; j < 7; j++)
            if(i != 2 || j != 3)
                drawEXIT(i * (fb.Var.xres / 5) + (fb.Var.xres / 20), j * (fb.Var.xres / 9) + (fb.Var.xres / 80), 253); // red
            else
                drawTEST(i * (fb.Var.xres / 5) + (fb.Var.xres / 20), j * (fb.Var.xres / 9) + (fb.Var.xres / 80), 251); // blue

    while(1)
    {
        read(ts_dev_fd, &ts_ev, sizeof(TS_EVENT));
        x = (int)(ts_ev.y * xscale - raw_min_x * xscale);
        y = (int)(ts_ev.x * yscale - raw_min_y * yscale);
        if(x >= x1 && x <= x2 && y >= y1 && y <= y2)
        {
            ezfb_clear(&fb);
            return 1;
        }
        if (ts_ev.pressure == 0)
            break;
    }

    ezfb_clear(&fb);
    return 0;
}

//############################################################################
void drawPlus(int xcoord, int ycoord, int cmap_index)
{
    int line_size = fb.Var.xres / 20 - 1;

    ezfb_put_line(&fb, xcoord - (line_size), /*  x1            */
                 ycoord              , /*  y1            */
                 xcoord + (line_size), /*  x2            */
                 ycoord              , /*  y2            */
                 cmap_index
           );

    ezfb_put_line(&fb, xcoord              ,
                 ycoord - (line_size),
                 xcoord              ,
                 ycoord + (line_size),
                 cmap_index
           );
}

//############################################################################
void drawX(int xcoord, int ycoord, int cmap_index)
{
    int line_size = fb.Var.xres / 20 - 1;

    ezfb_put_line(&fb, xcoord - line_size,
                 ycoord + line_size,
                 xcoord + line_size,
                 ycoord - line_size,
                 cmap_index
           );

    ezfb_put_line(&fb, xcoord - line_size,
                 ycoord - line_size,
                 xcoord + line_size,
                 ycoord + line_size,
                 cmap_index
           );
}

//############################################################################
void drawBeams(int xcoord, int ycoord, int cmap_index)
{
    ezfb_put_line(&fb, 0,
                 ycoord,
                 fb.Var.xres - 1,
                 ycoord,
                 cmap_index
           );

    ezfb_put_line(&fb, xcoord,
                 0,
                 xcoord,
                 fb.Var.yres - 1,
                 cmap_index
            );
}

//############################################################################
void drawTEST(int xcoord, int ycoord, int cmap_index)
{
    int scale = fb.Var.xres / 160;
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord,
                 ycoord,
                 xcoord,
                 ycoord +  6 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord,
                 ycoord,
                 xcoord + 21 * scale,
                 ycoord,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 21 * scale,
                 ycoord,
                 xcoord + 21 * scale,
                 ycoord +  6 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord,
                 ycoord +  6 * scale,
                 xcoord + 21 * scale,
                 ycoord +  6 * scale,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord + scale,
                 ycoord + scale,
                 xcoord + 5 * scale,
                 ycoord + scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 3 * scale,
                 ycoord + scale,
                 xcoord + 3 * scale,
                 ycoord + 5 * scale,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord +  6 * scale,
                 ycoord +      scale,
                 xcoord +  6 * scale,
                 ycoord +  5 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord +  6 * scale,
                 ycoord +      scale,
                 xcoord + 10 * scale,
                 ycoord +      scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord +  6 * scale,
                 ycoord +  3 * scale,
                 xcoord + 10 * scale,
                 ycoord +  3 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord +  6 * scale,
                 ycoord +  5 * scale,
                 xcoord + 10 * scale,
                 ycoord +  5 * scale,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord + 11 * scale,
                 ycoord +      scale,
                 xcoord + 15 * scale,
                 ycoord +      scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 11 * scale,
                 ycoord +  3 * scale,
                 xcoord + 15 * scale,
                 ycoord +  3 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 11 * scale,
                 ycoord +  5 * scale,
                 xcoord + 15 * scale,
                 ycoord +  5 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 11 * scale,
                 ycoord +      scale,
                 xcoord + 11 * scale,
                 ycoord +  3 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 15 * scale,
                 ycoord +  3 * scale,
                 xcoord + 15 * scale,
                 ycoord +  5 * scale,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord + 16 * scale,
                 ycoord +      scale,
                 xcoord + 20 * scale,
                 ycoord +      scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 18 * scale,
                 ycoord +      scale,
                 xcoord + 18 * scale,
                 ycoord +  5 * scale,
                 cmap_index
           );
}

//############################################################################
void drawEXIT(int xcoord, int ycoord, int cmap_index)
{
    int scale = fb.Var.xres / 160;

    ezfb_put_line(&fb, xcoord,
                 ycoord,
                 xcoord,
                 ycoord +  4 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord,
                 ycoord,
                 xcoord +  4 * scale,
                 ycoord,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord,
                 ycoord +  2 * scale,
                 xcoord +  4 * scale,
                 ycoord +  2 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord,
                 ycoord +  4 * scale,
                 xcoord +  4 * scale,
                 ycoord +  4 * scale,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord +  5 * scale,
                 ycoord,
                 xcoord +  9 * scale,
                 ycoord +  4 * scale,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord +  5 * scale,
                 ycoord +  4 * scale,
                 xcoord +  9 * scale,
                 ycoord,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord + 10 * scale,
                 ycoord,
                 xcoord + 10 * scale,
                 ycoord +  4 * scale,
                 cmap_index
           );
//-------------------------------------------------
    ezfb_put_line(&fb, xcoord + 11 * scale,
                 ycoord,
                 xcoord + 15 * scale,
                 ycoord,
                 cmap_index
           );
    ezfb_put_line(&fb, xcoord + 13 * scale,
                 ycoord,
                 xcoord + 13 * scale,
                 ycoord +  4 * scale,
                 cmap_index
           );
}

//############################################################################
void set_driver_parms()
{
#if TOUCH_SCREEN_CAL_DEBUG
    printf("UCB1200 PARMS BEING SET  :\n"                );
    printf("UCB1200_TS_SET_RAW_MAX_X : %d\n", raw_max_x);
    printf("UCB1200_TS_SET_RAW_MAX_Y : %d\n", raw_max_y);
    printf("UCB1200_TS_SET_RAW_MIN_X : %d\n", raw_min_x);
    printf("UCB1200_TS_SET_RAW_MIN_Y : %d\n", raw_min_y);
    printf("UCB1200_TS_SET_XY_SWAP   : %d\n", xy_swap  );
#endif
    ioctl(ts_dev_fd, UCB1200_TS_SET_RAW_MAX_X, raw_max_x);
    ioctl(ts_dev_fd, UCB1200_TS_SET_RAW_MAX_Y, raw_max_y);
    ioctl(ts_dev_fd, UCB1200_TS_SET_RAW_MIN_X, raw_min_x);
    ioctl(ts_dev_fd, UCB1200_TS_SET_RAW_MIN_Y, raw_min_y);
    ioctl(ts_dev_fd, UCB1200_TS_SET_XY_SWAP  , xy_swap  );

    return;
}

//############################################################################
///////////////////////////////////////////////////////////////////////////
//############################################################################
